/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.databinding;

import static org.fortiss.tooling.kernel.ui.util.DataBindingUtils.DECORATION_KEY;
import static org.fortiss.tooling.kernel.ui.util.DataBindingUtils.performCellTextBinding;

import org.eclipse.core.databinding.Binding;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.conversion.IConverter;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnViewer;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.fortiss.tooling.kernel.ui.extension.base.EditorBase;
import org.fortiss.tooling.kernel.ui.util.DataBindingUtils;

/**
 * <p>
 * Abstract class implementing text cell editing support with decorated cell editor. The binding is
 * created using {@link DataBindingUtils} and the information provided by sub-classes in the
 * abstract methods.
 * </p>
 * <P>
 * Decoration is in effect if either {@link #getTextValidator(Object)} returns a non-{@code null}
 * {@link IValidator}.
 * </p>
 * 
 * @author hoelzl
 */
public abstract class AbstractDecoratedStyledTextCellDatabindingEditingSupport
		extends AbstractStyledTextCellDatabindingEditingSupport {

	/** Constructor. */
	public AbstractDecoratedStyledTextCellDatabindingEditingSupport(
			EditorBase<? extends EObject> editor, ColumnViewer viewer,
			DataBindingContext bindingContext) {
		super(editor, viewer, bindingContext);
	}

	/** {@inheritDoc} */
	@Override
	protected void setupCellEditor(CellEditor cellEditor) {
		cellEditor.getControl().setData(DECORATION_KEY,
				new ControlDecoration(cellEditor.getControl(), SWT.LEFT | SWT.TOP));
	}

	/** {@inheritDoc} */
	@Override
	protected Binding[] createBinding(CellEditor cellEditor, ViewerCell cell, CellEditor editor,
			DataBindingContext context) {
		return performCellTextBinding(dbc, getText(cellEditor),
				getModelObservableValue(cell.getElement()), getModelToTextConverter(),
				getTextToModelConverter(), getTextValidator(cell.getElement()));
	}

	/** Sub-classes need to provide an {@link IObservableValue} for the model. */
	protected abstract IObservableValue<?> getModelObservableValue(Object model);

	/**
	 * Sub-classes need to provide an {@link IConverter} that converts the model into its textual
	 * representation.
	 */
	protected abstract IConverter getModelToTextConverter();

	/**
	 * Sub-classes need to provide an {@link IConverter} that converts a text into the model
	 * representation.
	 */
	protected abstract IConverter getTextToModelConverter();

	/**
	 * Sub-classes should overwrite to provide a text {@link IValidator}.
	 * 
	 * @param object
	 *            Object for which to provide text {@link IValidator}.
	 * 
	 * @return <code>null</code>, by default.
	 */
	protected IValidator getTextValidator(Object object) {
		return null;
	}

	/** Returns the {@link Control} to be used for data binding. Sub-classes may override. */
	protected Control getText(CellEditor cellEditor) {
		return cellEditor.getControl();
	}
}
