/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.databinding;

import org.eclipse.core.databinding.Binding;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnViewer;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationEvent;
import org.eclipse.jface.viewers.ColumnViewerEditorActivationListener;
import org.eclipse.jface.viewers.ColumnViewerEditorDeactivationEvent;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.widgets.Composite;
import org.fortiss.tooling.kernel.ui.util.DataBindingUtils;

/**
 * Abstract base class for inline text cell editors using data binding.
 * 
 * @author hoelzl
 */
public abstract class AbstractTextCellDatabindingEditingSupport extends EditingSupport {

	/** Cell editor. */
	protected CellEditor cellEditor;

	/** Data binding context. */
	protected final DataBindingContext dbc;

	/**
	 * Complex data bindings.
	 * 
	 * @see DataBindingUtils#performCellTextBinding(DataBindingContext,
	 *      org.eclipse.swt.widgets.Control, IObservableValue,
	 *      org.eclipse.core.databinding.conversion.IConverter,
	 *      org.eclipse.core.databinding.conversion.IConverter,
	 *      org.eclipse.core.databinding.validation.IValidator)
	 */
	private Binding[] bindings;

	/**
	 * Listener that destroys {@link Binding}s when the inline editor
	 * disappears.
	 */
	private final ColumnViewerEditorActivationListenerHelper activationListener =
			new ColumnViewerEditorActivationListenerHelper();

	/** Constructor. */
	public AbstractTextCellDatabindingEditingSupport(ColumnViewer viewer,
			DataBindingContext bindingContext) {
		super(viewer);
		this.dbc = bindingContext;
	}

	/** Override this method to create another cell editor. */
	protected CellEditor createCellEditor(ColumnViewer viewer) {
		CellEditor cellEditor = new TextCellEditor((Composite)viewer.getControl());
		setupCellEditor(cellEditor);
		return cellEditor;
	}

	/** Set up the new created {@link CellEditor} */
	protected abstract void setupCellEditor(CellEditor cellEditor);

	/** {@inheritDoc} */
	@Override
	public CellEditor getCellEditor(Object model) {
		cellEditor = createCellEditor(getViewer());
		return cellEditor;
	}

	/** {@inheritDoc} */
	@Override
	protected boolean canEdit(Object element) {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	protected final Object getValue(Object element) {
		// not needed
		return null;
	}

	/** {@inheritDoc} */
	@Override
	protected final void setValue(Object element, Object value) {
		// not needed
	}

	/** {@inheritDoc} */
	@Override
	protected void initializeCellEditorValue(CellEditor cellEditor, ViewerCell cell) {

		// reset cell editor, because null is not interpreted as empty string
		// the value will be set afterwards if it is not null
		this.cellEditor.setValue("");
		bindings = createBinding(cellEditor, cell, this.cellEditor, dbc);

		Assert.isTrue(bindings != null && bindings.length > 0 && bindings[0] != null,
				"Illegal implementation: no binding returned.");

		getViewer().getColumnViewerEditor().addEditorActivationListener(activationListener);
	}

	/**
	 * Creates the current bindings. Sub-classes need to return at least one
	 * binding. Furthermore, the first binding needs to be the binding that
	 * effectively stores the value to the model. This binding's
	 * {@link Binding#updateTargetToModel()} is called when the inline editor is
	 * closed. After that all the bindings returned here are disposed.
	 */
	protected abstract Binding[] createBinding(CellEditor cellEditor, ViewerCell cell,
			CellEditor editor, DataBindingContext context);

	/** {@inheritDoc} */
	@Override
	protected void saveCellEditorValue(CellEditor cellEditor, ViewerCell cell) {
		if(bindings != null && bindings.length > 0 && bindings[0] != null) {
			bindings[0].updateTargetToModel();
		}
	}

	/**
	 * This listener disposes the bindings as soon as the inline editor is
	 * deactivated. This assures that no binding created by
	 * {@link #createBinding(CellEditor, ViewerCell, CellEditor, DataBindingContext)} is left behind
	 * in the {@link DataBindingContext}.
	 */
	private class ColumnViewerEditorActivationListenerHelper
			extends ColumnViewerEditorActivationListener {

		/** {@inheritDoc} */
		@Override
		public void afterEditorActivated(ColumnViewerEditorActivationEvent event) {
			// do nothing
		}

		/** {@inheritDoc} */
		@Override
		public void afterEditorDeactivated(ColumnViewerEditorDeactivationEvent event) {

			if(bindings != null) {
				for(final Binding binding : bindings) {
					if(binding != null) {
						binding.dispose();
					}
				}
				bindings = null;
			}
			getViewer().getColumnViewerEditor().removeEditorActivationListener(this);
		}

		/** {@inheritDoc} */
		@Override
		public void beforeEditorActivated(ColumnViewerEditorActivationEvent event) {
			// do nothing
		}

		/** {@inheritDoc} */
		@Override
		public void beforeEditorDeactivated(ColumnViewerEditorDeactivationEvent event) {
			// do nothing
		}
	}
}
