/*-------------------------------------------------------------------------+
| Copyright 2018 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.databinding;

import static org.eclipse.core.databinding.validation.ValidationStatus.cancel;
import static org.eclipse.core.databinding.validation.ValidationStatus.ok;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParsePosition;

import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.core.runtime.IStatus;

/**
 * Validates if a given input is a valid {@link Number}.
 * 
 * @author barner
 */
public abstract class NumberValidator implements IValidator {

	/** The number format used in the current locale. */
	private final NumberFormat format;

	/** Character used to separate blocks of large numbers. */
	private Character groupingSeparator;

	/** Constructor. */
	protected NumberValidator(NumberFormat format) {
		this.format = format;
		if(format instanceof DecimalFormat) {
			DecimalFormat df = (DecimalFormat)format;
			groupingSeparator = df.getDecimalFormatSymbols().getGroupingSeparator();
		}
	}

	/** Returns the format name. */
	protected abstract String getFormatName();

	/** {@inheritDoc} */
	@Override
	public IStatus validate(Object value) {
		String input = value != null ? value.toString() : "";
		String trimmedInput = input.trim();
		if(trimmedInput.isEmpty()) {
			return ok();
		}
		if((groupingSeparator != null) && (trimmedInput.indexOf(groupingSeparator) != -1)) {
			return cancel("Grouping separator \'" + groupingSeparator + "\' not allowed");
		}

		ParsePosition p = new ParsePosition(0);
		format.parse(trimmedInput, p);
		int errorIndex = p.getErrorIndex();

		if(errorIndex == -1) {
			// In case the input has a valid prefix, the error index will not indicate a problem.
			// Therefore, check if the entire input was parsed.
			if(p.getIndex() < trimmedInput.length()) {
				errorIndex = p.getIndex();
			} else {
				return ok();
			}
		}

		int n = input.indexOf(trimmedInput) + errorIndex;
		return cancel(getFormatName() + " format validation failed at position " + n);
	}
}
