/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension;

import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.fortiss.tooling.kernel.service.base.IEObjectAware;
import org.fortiss.tooling.kernel.ui.extension.base.ModelElementHandlerBase;
import org.fortiss.tooling.kernel.ui.extension.base.NamedCommentedModelElementHandlerBase;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;

import javafx.scene.Node;

/**
 * A model element handler provides information about a specific type of model
 * elements like name, description, icon and certain kinds of sub-elements.
 * 
 * @see IModelElementHandlerService
 * @see ModelElementHandlerBase
 * @see NamedCommentedModelElementHandlerBase
 * 
 * @author hoelzl
 */
public interface IModelElementHandler<T extends EObject> extends IEObjectAware<T> {

	/** Returns the name of the model element (if any). */
	String getName(T element);

	/** Returns a description of the model element (if any). */
	String getDescription(T element);

	/** Returns the icon of the model element. */
	Image getIcon(T element);

	/** Returns image descriptor to be used as icon image. */
	ImageDescriptor getIconImageDescriptor(T element);

	/** Returns the JavaFX icon of the model element. */
	Node getFXIcon(T element);

	/**
	 * Returns the path to the overlay image for the icon. The result may be 'null' if no overlay
	 * shall be added.
	 */
	String getIconOverlayPath(T element);

	/**
	 * Returns all children acting as nodes, which are usually displayed as edit
	 * parts by graphical editors and entries in the navigator tree.
	 */
	List<EObject> getSubnodes(T element);

	/**
	 * Returns all children acting as connectors, which are usually displayed as
	 * edit parts in graphical editors, but NOT as entries in the navigator
	 * tree.
	 */
	List<EObject> getConnectors(T element);

	/**
	 * Returns all connections entering this element, which are usually
	 * displayed as edit parts in graphical editors.
	 */
	List<EObject> getIncomingConnections(T element);

	/**
	 * Returns all connections leaving this element, which are usually displayed
	 * as edit parts in graphical editors.
	 */
	List<EObject> getOutgoingConnections(T element);

	/**
	 * Returns all children acting as a (visible) specification, which are
	 * displayed as child entries in the navigator tree, but not as edit parts
	 * in graphical editors.
	 */
	List<EObject> getSpecifications(T element);

	/**
	 * Returns true if the underlying element should be hidden in the
	 * navigator's simplified non-expert view.
	 */
	boolean hiddenInNonExpertView();

	/**
	 * Returns the weight of the model element used for sorting of model elements in the navigator
	 * view. Ordering is performed in ascending order.
	 */
	int getNavigatorViewWeight(T element);

	/**
	 * Handles the open model element request by returning the object that
	 * should be passed to the model editor service.
	 */
	EObject handleOpenModelElementRequest(T element);

	/**
	 * Handle property section re-targeting. This method is called when adding
	 * property sections for this element. It may return a different EObject
	 * than this, but MUST NOT return <code>null</code>.
	 */
	EObject getPropertySectionRetargetElement(T element);
}
