/*-------------------------------------------------------------------------+
| Copyright 2022 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension;

import static java.util.Collections.emptyList;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.data.ContextMenuContextProvider;
import org.fortiss.tooling.kernel.ui.internal.views.NavigatorTreeContentProvider;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;

/**
 * Context menu entry to reveal an element in its editor.
 * 
 * @author schoepp
 */
public class RevealMenu implements IContextMenuContributor {

	/** String appearing in the context menu. */
	private static String MENU_NAME = "Reveal in editor";

	/** Declares where is located the icon to use for the menu contribution */
	private static String MENU_ICON = "icons/project.png";

	/** Content provider for navigating the model tree */
	private NavigatorTreeContentProvider treeNavigatorTreeContentProvider =
			new NavigatorTreeContentProvider();

	/** Returns the icon that is visible in the context menu for this entry. */
	protected ImageDescriptor getActionIcon() {
		return ToolingKernelUIActivator.getImageDescriptor(MENU_ICON);
	}

	/** {@link Action} to review an element in its editor. */
	protected class RevealAction extends Action {
		/** {@link EObject} selected by user */
		private final EObject element;

		/** {@link EObject} selected by user */
		private final EObject parentElement;

		/** Constructor. */
		public RevealAction(EObject parentElement, EObject element) {
			super(MENU_NAME, getActionIcon());
			this.parentElement = parentElement;
			this.element = element;
		}

		/** {@inheritDoc} */
		@Override
		public void run() {
			IModelEditorBindingService.getInstance().openInEditor(parentElement);
			IModelEditor<?> editor = IModelEditorBindingService.getInstance().getActiveEditor();
			if(editor == null || !IModelEditorBindingService.getInstance().isOpen(parentElement)) {
				// If opening the parent didn't work (e.g. because it doesn't have an editor),
				// open the element itself at least.
				IModelEditorBindingService.getInstance().openInEditor(element);
			} else {
				editor.navigateTo(element);
			}
		}
	}

	/** Predicate if there are ModelEditorBindings for {@code element}. */
	private boolean hasBindings(EObject element) {
		List<IModelEditorBinding<EObject>> bindings =
				IModelEditorBindingService.getInstance().getBindings(element);
		if(bindings.isEmpty()) {
			return false;
		}

		// get(0): Size of bindings is 1
		return !(bindings.size() == 1 && bindings.get(0) instanceof ModelEditorNotAvailableBinding);
	}

	/** {@inheritDoc} */
	@Override
	public List<IContributionItem> getContributedItems(EObject selection,
			ContextMenuContextProvider contextProvider) {

		EObject element = openModelElementRequest(selection);
		if(element == null || !INavigatorService.getInstance().isNavigatorView(contextProvider)) {
			return emptyList();
		}
		Object parent = treeNavigatorTreeContentProvider.getParent(selection);
		if(!(parent instanceof EObject)) {
			return emptyList();
		}
		EObject parentElement = openModelElementRequest((EObject)parent);
		if(parentElement == null) {
			return emptyList();
		}

		if(!hasBindings(element) && !hasBindings(parentElement)) {
			return emptyList();
		}

		List<IContributionItem> contributionItems = new ArrayList<IContributionItem>();
		contributionItems.add(new ActionContributionItem(new RevealAction(parentElement, element)));
		return contributionItems;
	}

	/** Find the model element for a selection in the model navigator */
	private EObject openModelElementRequest(EObject selection) {
		if(selection == null) {
			return null;
		}
		IModelElementHandler<EObject> handler =
				IModelElementHandlerService.getInstance().getModelElementHandler(selection);
		if(handler == null) {
			return null;
		}
		return handler.handleOpenModelElementRequest(selection);
	}

	/** {@inheritDoc} */
	@Override
	public String getMenuSectionID() {
		return null;
	}
}
