/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static java.util.Arrays.asList;
import static org.eclipse.emf.common.util.ECollections.unmodifiableEList;

import java.util.List;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.resource.ImageDescriptor;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.service.ITutorialUIService;

/**
 * JFace {@link Action}, which uses an {@link EObject} as target.
 * 
 * @author trachtenherz
 * @author hoelzl
 */
public abstract class EObjectActionBase<T extends EObject> extends Action {

	/** Factory interface for action instances. */
	public static interface EObjectActionFactory<T extends EObject> {
		/** Creates and returns an action instance for the given target. */
		EObjectActionBase<T> createAction(T target);
	}

	/** Stores the target elements. */
	private EList<T> targets;

	/** Constructor. */
	public EObjectActionBase(String text, ImageDescriptor image) {
		super(text, image);
	}

	/** Constructor. */
	public EObjectActionBase(String text) {
		this(text, null);
	}

	/** Constructor. */
	public EObjectActionBase(T target, String text, ImageDescriptor image) {
		this(text, image);
		refresh(target);
	}

	/** Constructor. */
	public EObjectActionBase(T target, String text) {
		this(target, text, null);
	}

	/** Sets the target object. */
	public synchronized void setTargets(List<T> newTargets) {
		if(targets == null) {
			targets = new BasicEList<T>(newTargets);
		} else {
			targets.clear();
			targets.addAll(newTargets);
		}
	}

	/** Sets the target object. */
	public void setTarget(T target) {
		setTargets(asList(target));
	}

	/** Returns the current target. */
	public T getTarget() {
		List<T> targets = getTargets();

		// For actions that cannot deal with multiple selections, only the first
		// item is picked, hence the get(0)
		return(targets.isEmpty() ? null : targets.get(0));
	}

	/** Returns the current targets. */
	public synchronized EList<T> getTargets() {
		return unmodifiableEList(targets);
	}

	/** Refreshes the enabled state for the current action target. */
	public final void refresh() {
		if(getTarget() == null) {
			setEnabled(false);
			return;
		}
		setEnabled(computeEnabled());
	}

	/** Returns whether the global enable state of copy/paste actions w.r.t. tutorials. */
	protected final boolean isGlobalCopyPasteActionsVisible() {
		return !ITutorialService.getInstance().isTutorialActive() ||
				ITutorialUIService.getInstance().globalCopyPasteActionsVisible();
	}

	/** Returns whether the global enable state of copy/paste actions w.r.t. tutorials. */
	protected final boolean isGlobalDeleteActionVisible() {
		return !ITutorialService.getInstance().isTutorialActive() ||
				ITutorialUIService.getInstance().globalDeleteActionVisible();
	}

	/** Returns whether the global enable state of undo/redo actions w.r.t. tutorials. */
	protected final boolean isGlobalUndoRedoActionsVisible() {
		return !ITutorialService.getInstance().isTutorialActive() ||
				ITutorialUIService.getInstance().globalUndoRedoActionsVisible();
	}

	/**
	 * Refreshes the action's target and enabled state.
	 * 
	 * @see #setTarget(EObject)
	 * @see #refresh()
	 */
	public final void refresh(T element) {
		setTarget(element);
		refresh();
	}

	/**
	 * Refreshes the action's targets and enabled state.
	 * 
	 * @see #setTarget(EObject)
	 * @see #refresh()
	 */
	public final void refresh(List<T> elements) {
		setTargets(elements);
		refresh();
	}

	/**
	 * Computes the enabled state of the action for the current target, which is
	 * not <code>null</code> when this method is called. The default
	 * implementation returns <code>true</code>.
	 */
	protected boolean computeEnabled() {
		return true;
	}
}
