/*-------------------------------------------------------------------------+
| Copyright 2018 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static org.fortiss.tooling.kernel.ui.util.SelectionUtils.checkAndPickFirst;
import static org.fortiss.tooling.kernel.utils.KernelModelElementUtils.runAsCommand;

import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.fortiss.tooling.kernel.ui.presentation.ModelElementLabelProvider;

/**
 * Base class for property sections for editing {@link EReference} lists.
 * 
 * @author barner
 */
public abstract class EReferenceListPropertySectionBase<I extends EObject, R extends EObject>
		extends EReferencePropertySectionBase<I, R> implements IListPropertySection<I, R> {

	/**
	 * Element that has been selected the combo box provided by the base class
	 * {@link EReferencePropertySectionBase}. It is added to the model using the "add" button.
	 */
	private R selectedElement;

	/**
	 * {@link Composite} providing a {@link ListViewer} and an add and a remove {@link Button} to
	 * edit {@link EReference} list.
	 */
	public static class ElementListViewerComposite<I extends EObject, R extends EObject>
			extends Composite {
		/** {@link ListViewer} to show edited {@link EReference} list. */
		private ListViewer listViewer;

		/** {@link Button} to add an element to the {@link #listViewer}. */
		private Button addButton;

		/** {@link Button} to remove an element from the {@link #listViewer}. */
		private Button removeButton;

		/** Underlying {@link IListPropertySection}. */
		private IListPropertySection<I, R> section;

		/** Constructor. */
		public ElementListViewerComposite(Composite parent, int style,
				TabbedPropertySheetWidgetFactory wf, IListPropertySection<I, R> section) {
			super(parent, style);
			this.section = section;

			GridLayout layout = new GridLayout(2, false);
			layout.marginWidth = 0;
			setLayout(layout);

			listViewer = new ListViewer(this, SWT.BORDER | SWT.V_SCROLL);
			GridData gd = new GridData(GridData.FILL_HORIZONTAL);
			gd.grabExcessHorizontalSpace = true;
			gd.heightHint = 48;
			listViewer.getList().setLayoutData(gd);
			listViewer.setContentProvider(new ArrayContentProvider());
			listViewer.setLabelProvider(new ModelElementLabelProvider());

			listViewer.addSelectionChangedListener(new ISelectionChangedListener() {
				/** {@inheritDoc} */
				@Override
				public void selectionChanged(SelectionChangedEvent event) {
					// refresh() does not work here. It is based on isRemoveButtonEnabled(), that
					// queries the selection of the 'listViewer' has not been updated yet. When this
					// listener is triggered, only the 'event' already contains the new selection.
					R selectedElement = getFirstSelectedElement(event.getSelection());
					removeButton.setEnabled(selectedElement != null && section
							.canRemoveModelListElement(section.getSectionInput(), selectedElement));
				}
			});

			Composite buttonComposite = wf.createComposite(this, SWT.NONE);
			buttonComposite.setLayout(new FillLayout(SWT.VERTICAL));

			addButton = wf.createButton(buttonComposite, "Add", SWT.NONE);
			addButton.addSelectionListener(new SelectionAdapter() {
				/** {@inheritDoc} */
				@Override
				public void widgetSelected(SelectionEvent e) {
					I input = section.getSectionInput();
					runAsCommand(input,
							() -> section.addModelListElement(input, section.getSelectedElement()));
					section.refresh();
				}
			});

			removeButton = wf.createButton(buttonComposite, "Remove", SWT.NONE);
			removeButton.addSelectionListener(new SelectionAdapter() {
				/** {@inheritDoc} */

				@Override
				public void widgetSelected(SelectionEvent e) {
					R selectedElement = getFirstSelectedElement(listViewer.getSelection());
					if(selectedElement != null) {
						I input = section.getSectionInput();
						runAsCommand(input,
								() -> section.removeModelListElement(input, selectedElement));
						section.refresh();

						List<R> elements = section.getModelListElements(input);
						if(!elements.isEmpty()) {
							// Select first element (which has been checked to exist)
							listViewer.setSelection(new StructuredSelection(elements.get(0)));
						}
					}
				}
			});
		}

		/** Provides the first selected element of the give {@link ISelection}. */
		@SuppressWarnings("unchecked")
		private R getFirstSelectedElement(ISelection selection) {
			return (R)checkAndPickFirst(selection, EObject.class);
		}

		/**
		 * <p>
		 * Predicate if the {@link #addButton} is currently enabled.
		 * </p>
		 * <p>
		 * <b>NB: </b>Derived classes may override but must return the conjunction of their result
		 * and this implementation.
		 * </p>
		 */
		protected boolean isAddButtonEnabled() {
			R selectedElement = section.getSelectedElement();
			return selectedElement != null && !section
					.getModelListElements(section.getSectionInput()).contains(selectedElement);
		}

		/**
		 * <p>
		 * Predicate if the {@link #removeButton} is currently enabled.
		 * </p>
		 * <p>
		 * <b>NB: </b>Derived classes may override but must return the conjunction of their result
		 * and this implementation.
		 * </p>
		 */
		protected boolean isRemoveButtonEnabled() {
			R selectedElement = getFirstSelectedElement(listViewer.getSelection());
			return selectedElement != null &&
					section.canRemoveModelListElement(section.getSectionInput(), selectedElement);
		}

		/** Refreshes the input and the enabled state of the controls. */
		public void refresh() {
			addButton.setEnabled(false);
			removeButton.setEnabled(false);

			I input = section.getSectionInput();
			if(input == null) {
				return;
			}

			List<R> elements = section.getModelListElements(input);
			listViewer.setInput(elements);
			addButton.setEnabled(isAddButtonEnabled());
			removeButton.setEnabled(isRemoveButtonEnabled());
		}
	}

	/** {@link Composite} providing a {@link ListViewer} and an add and remove Buttons. */
	private ElementListViewerComposite<I, R> listViewerComposite;

	/** The edited object (updated in {@link #setSectionInput(Object)}). */
	private I input = null;

	/** Constructor . */
	public EReferenceListPropertySectionBase(final String label) {
		super(label);
	}

	/** {@inheritDoc} */
	@Override
	public void createControls(Composite parent, TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);

		listViewerComposite =
				new ElementListViewerComposite<>(composite, SWT.NONE, getWidgetFactory(), this);
		createFormEntry(listViewerComposite, "");
	}

	// Implementation of IListPropertySection
	/** {@inheritDoc} */
	@Override
	public final R getSelectedElement() {
		return selectedElement;
	}

	/** {@inheritDoc} */
	@Override
	public final I getSectionInput() {
		return input;
	}

	// Implementation of PropertySectionBase
	/** {@inheritDoc} */
	@Override
	public void refresh() {
		super.refresh();
		listViewerComposite.refresh();
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	protected void setSectionInput(Object input) {
		super.setSectionInput(input);
		this.input = (I)input;
	}

	// Implementation of EReferencePropertySectionBase
	/** {@inheritDoc} */
	@Override
	protected final EObject getModelValue(I input) {
		List<R> elements = getValues(input);
		if(elements.contains(selectedElement)) {
			return selectedElement;
		}

		// In the currently selected element is not in the list of available elements, default to
		// the first element in the list
		selectedElement = elements.isEmpty() ? null : elements.get(0);
		return selectedElement;
	}

	/** {@inheritDoc} */
	@Override
	protected final void setModelValue(I input, R newValue) {
		// Store the element selected in the combobox (provided by the base class) as input
		// for possible "add" operation.
		selectedElement = newValue;
	}
}
