/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static org.fortiss.tooling.kernel.utils.KernelModelElementUtils.computeFullyQualifiedName;

import java.util.List;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.model.INamedElement;
import org.fortiss.tooling.kernel.service.IPersistencyService;

/**
 * Base class for a property section that allows to set an {@link EReference} using a
 * {@link ComboViewer}.
 * 
 * <code>I</code> is the type of the object from which you read/write the reference. Typically the
 * container of the reference (not of the referenced object but of the reference itself!)
 * 
 * <code>R</code> is the type of the object being referred to.
 * 
 * @author aravantinos, barner
 */

public abstract class EReferencePropertySectionBase<I extends EObject, R extends EObject>
		extends PropertySectionBase {

	/** Combo for component references. */
	private ComboViewer modelReferenceCombo;

	/** {@link ComboViewer} label. */
	private final String modelReferenceComboLabel;

	/**
	 * Flag to know whether the selection is made by the class or by the user.
	 * Needed because a "selected" event is triggered in all cases and we do not want to register
	 * any transaction if the change does not come from the user. Not so nice but the interface of
	 * the {@link ComboViewer} does not provide any tool allowing to do better.
	 */
	private boolean userSelection;

	/**
	 * Entry in {@link #modelReferenceCombo} representing that no model element has been selected,
	 * i.e. that the {@link EReference} will be set to {@code null}.
	 */
	private Object NONE = new Object();

	/** The edited object (updated in {@link #setSectionInput(Object)}). */
	private I input = null;

	/**
	 * Constructs a new {@link EReferencePropertySectionBase}.
	 * 
	 * @param modelReferenceComboLabel
	 *            The label to be used in front of the combobox.
	 */
	public EReferencePropertySectionBase(final String modelReferenceComboLabel) {
		this.modelReferenceComboLabel = modelReferenceComboLabel;
	}

	/** {@inheritDoc} */
	@Override
	public void createControls(Composite parent, TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);
		modelReferenceCombo = createDecoratedComboViewer(modelReferenceComboLabel);
		modelReferenceCombo.setContentProvider(new ArrayContentProvider());
		modelReferenceCombo.setLabelProvider(new ReferenceLabelProvider());
		modelReferenceCombo.setComparator(new ViewerComparator());
		modelReferenceCombo.addSelectionChangedListener(new ISelectionChangedListener() {
			@SuppressWarnings("unchecked")
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection sel = (IStructuredSelection)event.getSelection();
				final Object newValue = sel.getFirstElement();

				if(userSelection) {
					ITopLevelElement modelContext =
							IPersistencyService.getInstance().getTopLevelElementFor(input);
					modelContext.runAsCommand(() -> {
						if(newValue == NONE) {
							setModelValue(input, null);
						} else if(newValue instanceof EObject) {
							EObject oldValue = getModelValue(input);
							if(oldValue != null && oldValue.equals(newValue)) {
								return;
							}
							setModelValue(input, (R)newValue);
						}
					});
					refresh();
				}
			}
		});
	}

	/**
	 * Returns the array of values to be offered in the {@link ComboViewer}.
	 * 
	 * @param input
	 *            Input object from which the list of values to be offered should be
	 *            determined.
	 * 
	 * @return {@link List} of values to be offered in the {@link ComboViewer}.
	 */
	protected abstract List<R> getValues(I input);

	/**
	 * Returns from the model the current value of the {@link EReference} edited by this
	 * {@link EReferencePropertySectionBase}.
	 * 
	 * @param input
	 *            Input object from which the current value should be determined.
	 * 
	 * @return Current value (from the model) of the {@link EReference} edited by this
	 *         {@link EReferencePropertySectionBase}.
	 */
	protected abstract EObject getModelValue(I input);

	/**
	 * Sets the current value in the model (from the current selection in the {@link ComboViewer}).
	 * 
	 * @param input
	 *            Input {@link EObject} in which the new value should be set.
	 * 
	 * @param newValue
	 *            New value of {@link EReference} to be set in the model.
	 */
	protected abstract void setModelValue(I input, R newValue);

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	protected void setSectionInput(Object input) {
		this.input = (I)input;
	}

	/** {@inheritDoc} */
	@Override
	public void refresh() {
		super.refresh();
		modelReferenceCombo.getCombo().setEnabled(false);

		if(input == null) {
			return;
		}

		// Initialize list of offered model elements
		EList<Object> initialInput = new BasicEList<Object>();
		initialInput.add(NONE);
		modelReferenceCombo.setInput(initialInput);
		List<R> values = getValues(input);
		if(values == null || values.isEmpty()) {
			setSelection(null);
			return;
		}
		modelReferenceCombo.getCombo().setEnabled(true);
		modelReferenceCombo.add(values.toArray());

		// Set initial selection
		EObject value = getModelValue(input);
		setSelection(values.contains(value) ? value : null);
	}

	/** Sets the selection of the {@link #modelReferenceCombo} . */
	private void setSelection(EObject value) {
		userSelection = false;
		modelReferenceCombo.setSelection(new StructuredSelection(value == null ? NONE : value));
		userSelection = true;
	}

	/**
	 * Returns the label text to be displayed for a given element. May be overridden in subclasses.
	 * 
	 * @param element
	 *            Element for which the label text to be displayed in the {@link ComboViewer} should
	 *            be determined.
	 * 
	 * @return Label text to be displayed for a given element
	 */
	protected String getElementLabel(R element) {
		return element instanceof INamedElement
				? computeFullyQualifiedName((INamedElement)element, false) : null;
	}

	/** Label provider for {@link EReferencePropertySectionBase#modelReferenceCombo}. */
	private class ReferenceLabelProvider extends LabelProvider {
		/** {@inheritDoc} */
		@Override
		public String getText(Object element) {
			if(element == NONE) {
				return "(None)";
			}
			@SuppressWarnings("unchecked") String label = getElementLabel((R)element);
			return label != null ? label : "<unnamed>";
		}
	}
}
