/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.databinding.EMFDataBindingContext;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.TextStyledTextActionHandler;
import org.eclipse.ui.part.EditorPart;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.ui.extension.IModelEditor;
import org.fortiss.tooling.kernel.ui.extension.IModelEditorBinding;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.internal.editor.IActionContributingEditor;
import org.fortiss.tooling.kernel.ui.internal.editor.ModelElementEditorInput;

/**
 * Base implementation of model element editors returned by {@link IModelEditorBinding}s.
 * <p>
 * By default this implementation enables the model element library view.
 * 
 * @author hoelzlf
 */
public abstract class EditorBase<T extends EObject> extends EditorPart
		implements IActionContributingEditor, IModelEditor<T> {

	/**
	 * The object shown in this editor. This is valid as soon as
	 * {@link #init(IEditorSite, IEditorInput)} has been called.
	 */
	protected T editedObject;

	/** The model element handler to be used with the {@link #editedObject}. */
	protected IModelElementHandler<T> handler;

	/** Databinding context used for model changes. */
	protected final EMFDataBindingContext dbc = new EMFDataBindingContext();

	/** Stores the text action handler. */
	protected TextStyledTextActionHandler textStyledTextActionHandler;

	/** return the {@link TextStyledTextActionHandler} */
	public TextStyledTextActionHandler getTextStyledTextActionHandler() {
		return textStyledTextActionHandler;
	}

	/** {@inheritDoc} */
	@Override
	public T getEditedObject() {
		return editedObject;
	}

	/** {@inheritDoc} */
	@Override
	public void dispose() {
		if(textStyledTextActionHandler != null) {
			textStyledTextActionHandler.dispose();
		}
		dbc.dispose();
		super.dispose();
	}

	/**
	 * Returns the list of visible model element classes. This list is used to
	 * determine, which elements are to be included in the model element
	 * library. Sub-classes should override the default behavior, which returns
	 * the class of the {@link #editedObject}.
	 */
	@SuppressWarnings("unchecked")
	public Class<? extends EObject>[] getVisibleEObjectTypes() {
		return new Class[] {editedObject.getClass()};
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		if(!(input instanceof ModelElementEditorInput)) {
			throw new PartInitException("Expected input of type " + ModelElementEditorInput.class);
		}
		ModelElementEditorInput meInput = (ModelElementEditorInput)input;

		editedObject = (T)meInput.getModelElement();
		if(editedObject == null) {
			throw new PartInitException("Missing model element!");
		}

		handler = (IModelElementHandler<T>)meInput.getModelElementHandler();
		if(handler == null) {
			throw new PartInitException("Missing model element handler!");
		}

		setSite(site);
		setInput(input);

		setPartName(handler.getName(editedObject));
		setContentDescription(handler.getDescription(editedObject));
	}

	/** Runs the given {@link Runnable} in the emfStore editing domain. */
	protected final void executeCommand(Runnable runner) {
		ICommandStackService.getInstance().runAsCommand(editedObject, runner);
	}

	/** {@inheritDoc} */
	@Override
	public final boolean isDirty() {
		return ICommandStackService.getInstance().isDirty(editedObject);
	}

	/** {@inheritDoc} */
	@Override
	public final void doSave(IProgressMonitor monitor) {
		ICommandStackService.getInstance().doSave(editedObject, monitor);
	}

	/** {@inheritDoc} */
	@Override
	public final boolean isSaveAsAllowed() {
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public final void doSaveAs() {
		// Saving is handled automatically by persistency service
	}

	/** {@inheritDoc} */
	@Override
	public void registerGlobalActions(IActionBars bars) {
		textStyledTextActionHandler = new TextStyledTextActionHandler(bars);
		addTextStyledText(textStyledTextActionHandler);
	}

	/**
	 * Override this method to pass the given handler to any {@link Text} or {@link StyledText} that
	 * you have.
	 */
	protected void addTextStyledText(
			@SuppressWarnings("unused") TextStyledTextActionHandler textStyledTextActionHandler) {
		// do nothing
	}

	/** {@inheritDoc} */
	@Override
	public void unregisterGlobalActions(IActionBars bars) {
		// nothing to do
	}

	/** {@inheritDoc} */
	@Override
	public void setFocus() {
		// nothing to do
	}

	/** {@inheritDoc} */
	@Override
	public void navigateTo(EObject element) {
		// nothing to do
	}

	/** {@inheritDoc} */
	@Override
	public void setHighlight(EObject element, boolean highlighted) {
		// nothing to do
	}

	/** {@inheritDoc} */
	@Override
	public void clearAllHighlights() {
		// nothing to do
	}
}
