/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static org.fortiss.tooling.base.utils.SystemUtils.isLinuxPlatform;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;

import javafx.beans.InvalidationListener;
import javafx.beans.Observable;
import javafx.embed.swt.FXCanvas;
import javafx.scene.Parent;
import javafx.scene.Scene;

/**
 * Base implementation of model element editors which are using JavaFX.
 * 
 * @author hoelzlf
 */
public abstract class FXEditorBase<T extends EObject> extends EditorBase<T>
		implements ISelectionProvider {
	/** The list of {@link ISelectionChangedListener}s. */
	private final List<ISelectionChangedListener> selectionListeners = new ArrayList<>();

	/** The canvas showing the JavaFX scene. */
	private FXCanvas canvas;

	/** The JavaFX scene. */
	private Scene scene;

	/** {@inheritDoc} */
	@Override
	public final void createPartControl(Composite parent) {
		canvas = new FXCanvas(parent, SWT.NONE);
		scene = new Scene(createSceneRoot());
		canvas.setScene(scene);
		getSite().setSelectionProvider(this);

		// Work around a bug in FXCanvas or SWT on Linux, where
		// scrolling does not work.
		// On Linux, FXCanvas receives a start gesture event when
		// the mouse is clicked. It disables scrolling while the
		// gesture is ongoing. However, the gesture end event never
		// arrives. As a result, scrolling is disabled after the first
		// mouse click. This work-around simply filters out all gesture
		// events, which should not be used in the FX part currently.
		if(isLinuxPlatform()) {
			canvas.getDisplay().addFilter(SWT.Gesture, e -> {
				e.type = SWT.NONE;
			});
		}
	}

	/** Creates the root node of the scene. */
	protected abstract Parent createSceneRoot();

	/** {@inheritDoc} */
	@Override
	final public void navigateTo(EObject element) {
		runWhenSceneReady(() -> {
			navigateToFX(element);
		});
	}

	/**
	 * Navigate to an element in the JavaFX scene.
	 * 
	 * @param element
	 *            The element to be shown in the editor.
	 */
	protected void navigateToFX(EObject element) {
		// default does nothing
	}

	/**
	 * Run the given runnable once the scene is ready.
	 * 
	 * An FXCanvas takes some time to initialize the embedded JavaFX scene.
	 * The scene can be resized more than once before it has the right size.
	 */
	private void runWhenSceneReady(Runnable runnable) {
		if(scene == null) {
			return;
		}
		Runnable layoutAndRun = () -> {
			Parent root = scene.getRoot();
			if(root != null) {
				root.layout();
			}
			runnable.run();
		};
		if(sceneSizeReady()) {
			layoutAndRun.run();
		} else {
			final InvalidationListener listener = new InvalidationListener() {
				@Override
				public void invalidated(Observable observable) {
					if(sceneSizeReady()) {
						scene.heightProperty().removeListener(this);
						scene.widthProperty().removeListener(this);
						layoutAndRun.run();
					}
				}
			};
			scene.heightProperty().addListener(listener);
			scene.widthProperty().addListener(listener);
		}
	}

	/** Check if the sizes of canvas and scene match. */
	private boolean sceneSizeReady() {
		Rectangle canvasBounds = canvas.getBounds();
		double sceneHeight = scene.getHeight();
		double sceneWidth = scene.getWidth();
		return canvasBounds.height == sceneHeight && canvasBounds.width == sceneWidth;
	}

	/** {@inheritDoc} */
	@Override
	public final void addSelectionChangedListener(ISelectionChangedListener listener) {
		if(!selectionListeners.contains(listener)) {
			selectionListeners.add(listener);
		}
	}

	/** {@inheritDoc} */
	@Override
	public final void removeSelectionChangedListener(ISelectionChangedListener listener) {
		selectionListeners.remove(listener);
	}

	/** {@inheritDoc} */
	@Override
	public final void setSelection(ISelection selection) {
		SelectionChangedEvent evt = new SelectionChangedEvent(this, selection);
		for(ISelectionChangedListener scl : selectionListeners) {
			scl.selectionChanged(evt);
		}
	}

	/** {@inheritDoc} */
	@Override
	public ISelection getSelection() {
		return StructuredSelection.EMPTY;
	}
}
