/*-------------------------------------------------------------------------+
| Copyright 2019 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static java.util.stream.Collectors.toList;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IEditorPart;
import org.fortiss.tooling.common.ui.javafx.lwfxef.DiagramViewer;
import org.fortiss.tooling.common.ui.javafx.lwfxef.DiagramViewerSelection;
import org.fortiss.tooling.common.ui.javafx.lwfxef.change.Change;
import org.fortiss.tooling.common.ui.javafx.lwfxef.controller.IControllerFactory;
import org.fortiss.tooling.common.ui.javafx.lwfxef.model.IModelFactory;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.visual.IVisualFactory;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.ui.extension.IModelEditorBinding;
import org.fortiss.tooling.kernel.ui.extension.base.factory.DelegatingControllerFactory;
import org.fortiss.tooling.kernel.ui.extension.base.factory.DelegatingModelFactory;
import org.fortiss.tooling.kernel.ui.extension.base.factory.DelegatingVisualFactory;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;

import javafx.scene.Parent;

/**
 * Base class for JavaFX-based {@link IEditorPart}s that display diagrams of {@link EObject}s.
 * 
 * @author hoelzl
 * @author diewald
 */
public abstract class LWFXEFEditorBase<T extends EObject> extends FXEditorBase<T> {
	/** The diagram viewer. */
	protected DiagramViewer viewer;

	/** References the delegating {@link IModelFactory} of this JavaFX {@link IEditorPart}. */
	private IModelFactory delegatingModelFactory;

	/** References the delegating {@link IVisualFactory} of this JavaFX {@link IEditorPart}. */
	private IVisualFactory delegatingVisualFactory;

	/** References the delegating {@link IControllerFactory} of this JavaFX {@link IEditorPart}. */
	private IControllerFactory delegatingControllerFactory;

	/**
	 * {@link IModelEditorBinding} which this JavaFX {@link IEditorPart} should use to look-up its
	 * {@link #delegatingModelFactory}, {@link #delegatingVisualFactory}, and
	 * {@link #delegatingControllerFactory}.
	 */
	private IModelEditorBinding<EObject> editorBinding;

	/** {@inheritDoc} */
	@Override
	protected final Parent createSceneRoot() {
		constructMVCFactories();

		viewer = new DiagramViewer(delegatingModelFactory, delegatingVisualFactory,
				delegatingControllerFactory, cb -> modelSelected(), chg -> applyModelChange(chg));
		customizeViewer();
		return viewer.getVisualNode();
	}

	/**
	 * Constructs delegating factories that wrap all factories registered by
	 * {@link IModelEditorBinding}s for the edited type. Delegating factories are used for extending
	 * editors of the same type. If an original {@link IModelEditorBinding} (i.e., the binding that
	 * was used to construct the editor) exists for this
	 * {@link IEditorPart} it will be preferred.
	 */
	protected void constructMVCFactories() {
		IModelEditorBindingService imebs = IModelEditorBindingService.getInstance();
		List<IModelEditorBinding<EObject>> bindings = imebs.getBindings(editedObject);

		// Get all compatible bindings (unless it is the original binding)
		bindings.removeIf(b -> {
			boolean isNotLWFXEFEditor = ((b.getEditorClass(editedObject) != null) &&
					(!LWFXEFEditorBase.class.isAssignableFrom(b.getEditorClass(editedObject))));
			boolean isPreferredEditorBinding = (editorBinding != null) && (b == editorBinding);

			return isNotLWFXEFEditor || isPreferredEditorBinding;
		});

		// Prefer the original binding
		if(editorBinding != null) {
			bindings.add(0, editorBinding);
		}

		List<Class<? extends IModelFactory>> modelFactories = bindings.stream()
				.map(b -> b.getModelFactory()).filter(Objects::nonNull).collect(toList());
		List<Class<? extends IVisualFactory>> visualFactories = bindings.stream()
				.map(b -> b.getVisualFactory()).filter(Objects::nonNull).collect(toList());
		List<Class<? extends IControllerFactory>> controllerFactories = bindings.stream()
				.map(b -> b.getControllerFactory()).filter(Objects::nonNull).collect(toList());

		delegatingModelFactory = new DelegatingModelFactory(modelFactories, editedObject);
		delegatingVisualFactory = new DelegatingVisualFactory(visualFactories);
		delegatingControllerFactory = new DelegatingControllerFactory(controllerFactories);
	}

	/** Customize the viewer after it is created. The default implementation changes nothing. */
	protected void customizeViewer() {
		// the default does nothing
	}

	/** Called when a model element is selected in the diagram viewer. */
	protected void modelSelected() {
		getSite().getSelectionProvider().setSelection(createSelection());
	}

	/** Called when some change to the model happens. */
	protected void applyModelChange(Change chg) {
		ICommandStackService.getInstance().runAsCommand(editedObject, () -> chg.applyChange());
	}

	/** {@inheritDoc} */
	@Override
	public ISelection getSelection() {
		return createSelection();
	}

	/** Creates an {@link IStructuredSelection} from {@link DiagramViewerSelection}. */
	private IStructuredSelection createSelection() {
		DiagramViewerSelection vsel = viewer.getSelection();
		if(vsel.isEmpty()) {
			return StructuredSelection.EMPTY;
		}
		List<Object> selList = new ArrayList<>(vsel.getSecondarySelections().size() + 1);
		selList.add(vsel.getPrimarySelection().getModel());
		for(IMVCBundle sec : vsel.getSecondarySelections()) {
			selList.add(sec.getModel());
		}
		return new StructuredSelection(selList);
	}

	/** {@inheritDoc} */
	@Override
	public void navigateToFX(EObject element) {
		viewer.scrollToComponent(element);
	}

	/** Setter for {@link #editorBinding}. */
	public void setEditorBinding(IModelEditorBinding<EObject> editorBinding) {
		this.editorBinding = editorBinding;
	}
}
