/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static java.util.Collections.emptyList;
import static org.eclipse.jface.resource.ResourceLocator.imageDescriptorFromBundle;
import static org.fortiss.tooling.common.ui.javafx.util.GraphicUtils.getFXImageFromURI;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.warning;
import static org.fortiss.tooling.kernel.utils.ResourceUtils.getResourceURI;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DecorationOverlayIcon;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;

import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.SnapshotParameters;
import javafx.scene.effect.BlendMode;
import javafx.scene.image.ImageView;
import javafx.scene.image.WritableImage;

/**
 * Base implementation for {@link IModelElementHandler}s.
 * <br>
 * {@link ModelEditorBindingBase} retrieves model element icons from file resources in the plugins.
 * Sub-classes must implement {@link #getPluginId()} and {@link #getIconPath(EObject)} and may
 * override the other non-final methods of {@link IModelElementHandler}.
 * <br>
 * By default the base implementation returns empty lists for the respective methods. Open editor
 * requests are handled by returning the model element itself.
 * 
 * @author hoelzl
 */
public abstract class ModelElementHandlerBase<T extends EObject>
		implements IModelElementHandler<T> {
	/** Stores the icon image. */
	private Map<ImageDescriptor, Image> iconImages = new HashMap<ImageDescriptor, Image>();

	/** Caches the known elements without icon. */
	private Set<String> elementsWithOutIcons = new HashSet<>();

	/** {@inheritDoc} */
	@Override
	public String getDescription(T element) {
		return "";
	}

	/**
	 * Returns the ID of the plugin used by this model element handler to provide icons for model
	 * elements.
	 */
	protected abstract String getPluginId();

	/** Returns the plugin-relative path of the icon to be displayed for the given element. */
	protected String getIconPath(@SuppressWarnings("unused") T element) {
		// By default, no icon is given/wanted, which is represented by an empty string to
		// distinguish it from an actual error causing null.
		return "";
	}

	/** {@inheritDoc} */
	@Override
	public String getIconOverlayPath(T element) {
		// This default implementation does not add an overlay.
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public final ImageDescriptor getIconImageDescriptor(T element) {
		String pluginId = getPluginId();
		String iconPath = getIconPath(element);
		if(pluginId == null || iconPath == null) {
			String elementTypeName =
					element != null ? element.getClass().getCanonicalName() : "<null>";
			if(!elementsWithOutIcons.contains(elementTypeName)) {
				String pluginIdName = pluginId != null ? pluginId : "<null>";
				String iconPathName = iconPath != null ? iconPath : "<null>";
				warning(ToolingKernelUIActivator.getDefault(),
						"Failed to load icon for " + elementTypeName + ". Plugin ID: " +
								pluginIdName + ", icon path: " + iconPathName);
				elementsWithOutIcons.add(elementTypeName);
			}
			return null;
		}
		if(iconPath.isBlank()) {
			// Case: no icon is wanted (no warning needed).
			return null;
		}
		return imageDescriptorFromBundle(pluginId, iconPath).orElse(null);
	}

	/** {@inheritDoc} */
	@Override
	public final Image getIcon(T element) {
		ImageDescriptor descr = getIconImageDescriptor(element);
		if(descr == null) {
			return null;
		}
		Image iconImage = iconImages.get(descr);
		if(iconImage == null) {
			iconImage = descr.createImage();
			iconImages.put(descr, iconImage);
		}

		ImageDescriptor overlay = getIconOverlay(element);
		if(overlay != null) {
			Point resultImgSize = new Point(16, 16);
			ImageDescriptor[] overlaysArray =
					new ImageDescriptor[] {null, null, null, overlay, null};
			DecorationOverlayIcon decorated =
					new DecorationOverlayIcon(iconImage, overlaysArray, resultImgSize);
			return decorated.createImage();
		}

		return iconImage;
	}

	/** {@inheritDoc} */
	@Override
	public final Node getFXIcon(T element) {
		String pluginId = getPluginId();

		String pluginIdName = pluginId != null ? pluginId : "<null>";
		String elementTypeName = element != null ? element.getClass().getCanonicalName() : "<null>";

		// Get the Image for the icon itself
		String iconPath = getIconPath(element);
		if(iconPath != null && iconPath.isBlank()) {
			// Case: no icon is wanted (no warning needed).
			return null;
		}

		String iconUri;
		try {
			iconUri = getResourceURI(pluginId, iconPath);
		} catch(Exception e) {
			iconUri = null;
		}
		if(iconUri == null) {
			String iconPathName = iconPath != null ? iconPath : "<null>";
			warning(ToolingKernelUIActivator.getDefault(),
					"Failed to load FX icon for " + elementTypeName + ". Plugin ID: " +
							pluginIdName + ", icon path: " + iconPathName);
			return null;
		}
		javafx.scene.image.Image icon = getFXImageFromURI(iconUri);

		// Get the Image for the overlay, if there is a path specified
		String overlayPath = getIconOverlayPath(element);
		if(overlayPath != null && !overlayPath.isEmpty()) {
			String overlayUri;

			try {
				overlayUri = getResourceURI(pluginId, overlayPath);
			} catch(Exception e) {
				overlayUri = null;
			}
			if(overlayUri == null) {
				warning(ToolingKernelUIActivator.getDefault(),
						"Failed to load FX overlay for " + elementTypeName + ". Plugin ID: " +
								pluginIdName + ", overlay path: \"" + overlayPath + "\n");
			}
			javafx.scene.image.Image ovl = getFXImageFromURI(overlayUri);

			// Blend the icon with the overlay
			ImageView iconView = new ImageView(icon);
			ImageView ovlView = new ImageView(ovl);
			ovlView.setBlendMode(BlendMode.SRC_OVER);
			Group blend = new Group(iconView, ovlView);

			WritableImage blendedIcon = blend.snapshot(new SnapshotParameters(), null);

			return new ImageView(blendedIcon);
		}

		return new ImageView(icon);
	}

	/** {@inheritDoc} */
	@Override
	public List<EObject> getSubnodes(T element) {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public List<EObject> getConnectors(T element) {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public List<EObject> getIncomingConnections(T element) {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public List<EObject> getOutgoingConnections(T element) {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public List<EObject> getSpecifications(T element) {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public boolean hiddenInNonExpertView() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 * <P>
	 * The default returns 1000000.
	 */
	@Override
	public int getNavigatorViewWeight(T element) {
		return 1000000;
	}

	/** {@inheritDoc} */
	@Override
	public EObject handleOpenModelElementRequest(T element) {
		return element;
	}

	/** {@inheritDoc} */
	@Override
	public EObject getPropertySectionRetargetElement(T element) {
		return element;
	}

	/**
	 * Returns the overlay, that is applied on top of the icon of the given element.
	 * This overlay is used in {@link #getIcon(EObject)} to decorate the image returned by
	 * {@link #getIconImageDescriptor(EObject)}.
	 * 
	 * @param element
	 *            The element for which its icon shall be decorated with the overlay.
	 * @return The decoration overlay for element's icon.
	 */
	public ImageDescriptor getIconOverlay(T element) {
		return null;
	}
}
