/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base;

import static com.ibm.icu.text.NumberFormat.getNumberInstance;
import static org.eclipse.core.databinding.conversion.text.NumberToStringConverter.fromBigDecimal;
import static org.eclipse.core.databinding.conversion.text.NumberToStringConverter.fromDouble;
import static org.eclipse.core.databinding.conversion.text.NumberToStringConverter.fromInteger;
import static org.eclipse.core.databinding.conversion.text.StringToNumberConverter.toBigDecimal;
import static org.eclipse.core.databinding.conversion.text.StringToNumberConverter.toDouble;
import static org.eclipse.core.databinding.conversion.text.StringToNumberConverter.toInteger;
import static org.fortiss.tooling.kernel.ui.databinding.FloatValidator.FLOAT_VALIDATOR;
import static org.fortiss.tooling.kernel.ui.databinding.IntValidator.INT_VALIDATOR;
import static org.fortiss.tooling.kernel.ui.databinding.NumberPositiveValidator.NUMBER_POSITIVE_VALIDATOR;
import static org.fortiss.tooling.kernel.ui.databinding.NumberPositiveZeroValidator.NUMBER_POSITIVE_ZERO_VALIDATOR;
import static org.fortiss.tooling.kernel.ui.util.DataBindingUtils.DECORATION_KEY;
import static org.fortiss.tooling.kernel.ui.util.DataBindingUtils.performComplexTextBinding;
import static org.fortiss.tooling.kernel.ui.util.WidgetsFactory.createTextWithUndo;

import java.math.BigDecimal;

import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.databinding.EMFDataBindingContext;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.EditPart;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.ComboViewer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.fortiss.tooling.kernel.model.FortissToolingKernelPackage;

import com.ibm.icu.text.NumberFormat;

/**
 * Base class for property sections, dealing with setting the input and
 * providing some utility methods.
 * 
 * @author hoelzl
 */
public abstract class PropertySectionBase extends AbstractPropertySection {

	/** Adapter to synchronize property sheet title. */
	private final class TitleSyncAdapter extends AdapterImpl {
		/** {@inheritDoc} */
		@Override
		public void notifyChanged(org.eclipse.emf.common.notify.Notification msg) {
			if(msg.getEventType() == Notification.SET &&
					msg.getFeature() == FortissToolingKernelPackage.Literals.INAMED_ELEMENT__NAME) {
				// Model change could be programmatic: wrap the update into asyncExec().
				Display.getDefault().asyncExec(() -> {
					// Updates the tabbed property sheets title (no other API available)
					tabbedPropertySheetPage.labelProviderChanged(null);
				});
			}
			// System.out.println(msg.getEventType() + " @ " + msg.getNotifier());
		}

		/** Removes itself. */
		public void remove() {
			Notifier target = getTarget();
			if(target != null) {
				target.eAdapters().remove(this);
			}
		}
	}

	/** Label width. */
	public static final int PROPERTIES_LABEL_WIDTH = 150;

	/** The main composite for the controls. */
	protected Composite composite;

	/** Stores the data binding context used. */
	protected EMFDataBindingContext dbc = new EMFDataBindingContext();

	/** Underlying {@link TabbedPropertySheetPage}. */
	private TabbedPropertySheetPage tabbedPropertySheetPage;

	/** Adapter to synchronize property sheet title. */
	private TitleSyncAdapter titleSyncAdapter = new TitleSyncAdapter();

	/** {@inheritDoc} */
	@Override
	public void createControls(Composite parent, TabbedPropertySheetPage tabbedPropertySheetPage) {
		super.createControls(parent, tabbedPropertySheetPage);
		this.tabbedPropertySheetPage = tabbedPropertySheetPage;

		composite = getWidgetFactory().createFlatFormComposite(parent);
		composite.setLayout(new GridLayout(2, false));

		/*
		 * TODO: workaround for bug #1863@af3-redmine & #383750@eclipse-bug-tracker
		 * if the first control in a multi-tab property view is a list, the bug will not be
		 * triggered.
		 */
		getWidgetFactory().createList(composite, SWT.NONE).setLayoutData(new GridData(1, 1));
		getWidgetFactory().createLabel(composite, "", SWT.NONE).setLayoutData(new GridData(1, 1));
	}

	/** {@inheritDoc} */
	@Override
	public void dispose() {
		dbc.dispose();
		dbc = null;
		super.dispose();
	}

	/** {@inheritDoc} */
	@Override
	public final void setInput(IWorkbenchPart part, ISelection selection) {
		super.setInput(part, selection);

		if(!(selection instanceof IStructuredSelection)) {
			return;
		}
		Object object = ((IStructuredSelection)selection).getFirstElement();
		if(object instanceof EditPart) {
			object = ((EditPart)object).getModel();
		}
		if(object instanceof EObject) {
			addTitleSyncAdapter((EObject)object);
		}

		setSectionInput(object);
	}

	/**
	 * Adds the {@link #titleSyncAdapter} to the given {@link EObject}, and undoes any
	 * possible previous registration.
	 */
	protected void addTitleSyncAdapter(EObject eObj) {
		titleSyncAdapter.remove();
		eObj.eAdapters().add(titleSyncAdapter);
	}

	/** Disables all children controls. */
	protected void disableControls() {
		for(Control c : composite.getChildren()) {
			c.setEnabled(false);
		}
	}

	/** Enables all children controls. */
	protected void enableControls() {
		for(Control c : composite.getChildren()) {
			c.setEnabled(true);
		}
	}

	/** This is used to set the single section input. The parameter may be null! */
	protected abstract void setSectionInput(Object input);

	/**
	 * Creates a {@link Text} control which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected Text createDecoratedText(String value, int style) {
		Text text = createTextWithUndo(getWidgetFactory(), composite, value, style);
		text.setData(DECORATION_KEY, new ControlDecoration(text, SWT.LEFT | SWT.TOP));
		return text;
	}

	/**
	 * Creates a {@link Text} control which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected Text createDecoratedText(String value) {
		return createDecoratedText(value, SWT.NONE);
	}

	/**
	 * Creates a form text field, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected Text createDecoratedFormText(String labelText, int style) {
		Text text = createFormText(labelText, style);
		text.setData(DECORATION_KEY, new ControlDecoration(text, SWT.LEFT | SWT.TOP));
		return text;
	}

	/**
	 * Creates a form text field, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected Text createDecoratedFormText(String labelText) {
		return createDecoratedFormText(labelText, SWT.NONE);
	}

	/** Creates form text field. */
	protected Text createFormText(String labelText, int style) {
		Text text = createTextWithUndo(getWidgetFactory(), composite, "", style);
		createFormEntry(text, labelText);
		return text;
	}

	/** Creates form text field. */
	protected Text createFormText(String labelText) {
		return createFormText(labelText, SWT.NONE);
	}

	/**
	 * Creates a combo widget, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected ComboViewer createDecoratedComboViewer(String labelCombo) {
		ComboViewer combo = createFormComboViewer(labelCombo);
		combo.setData(DECORATION_KEY, new ControlDecoration(combo.getCombo(), SWT.LEFT | SWT.TOP));
		return combo;
	}

	/** Creates form combo field. */
	protected ComboViewer createFormComboViewer(String labelCombo) {
		ComboViewer combo = new ComboViewer(composite, SWT.READ_ONLY);
		createFormEntry(combo.getCombo(), labelCombo);
		return combo;
	}

	/**
	 * Creates a combo widget, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected CCombo createDecoratedCombo(String labelCombo) {
		CCombo combo = createFormCombo(labelCombo);
		combo.setData(DECORATION_KEY, new ControlDecoration(combo, SWT.LEFT | SWT.TOP));
		return combo;
	}

	/** Creates form combo field. */
	protected CCombo createFormCombo(String labelCombo) {
		CCombo combo = getWidgetFactory().createCCombo(composite);
		createFormEntry(combo, labelCombo);
		return combo;
	}

	/**
	 * Creates a list widget, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected TreeViewer createDecoratedTree(String labeltree) {
		TreeViewer treeViewer = createFormTree(labeltree);
		treeViewer.setData(DECORATION_KEY,
				new ControlDecoration(treeViewer.getTree(), SWT.LEFT | SWT.TOP));
		return treeViewer;
	}

	/** Creates form list field. */
	protected TreeViewer createFormTree(String labeltree) {
		TreeViewer treeViewer = new TreeViewer(composite);
		createFormEntry(treeViewer.getTree(), labeltree);
		return treeViewer;
	}

	/**
	 * Creates a combo widget, which has a {@link ControlDecoration} attached.
	 * The decoration is stored as widget data for convenience.
	 */
	protected Button createDecoratedButton(String labelButton, Control control) {
		Composite parent = getWidgetFactory().createComposite(composite);
		parent.setLayout(new GridLayout(2, false));
		parent.setLayoutData(control.getLayoutData());
		control.setParent(parent);

		Button button = getWidgetFactory().createButton(parent, labelButton, SWT.PUSH);
		GridDataFactory.defaultsFor(button).hint(labelButton.length() * 10, SWT.DEFAULT)
				.align(SWT.END, SWT.BEGINNING).applyTo(button);
		GridData gd = GridDataFactory.defaultsFor(control).create();
		gd.widthHint = gd.widthHint - labelButton.length() * 10;

		control.setLayoutData(gd);
		return button;
	}

	/** Creates a form entry and returns the {@link Label}. */
	protected Label createFormEntry(Control control, String labelName) {
		labelName = labelName.trim();
		if(labelName.endsWith(":")) {
			labelName = labelName.replaceAll(":$", "");
		}

		Label label = getWidgetFactory().createLabel(composite, labelName);
		label.moveAbove(control);
		GridDataFactory.defaultsFor(label).hint(PROPERTIES_LABEL_WIDTH, SWT.DEFAULT)
				.align(SWT.BEGINNING, SWT.BEGINNING).applyTo(label);
		GridData gd = GridDataFactory.defaultsFor(control).create();
		gd.heightHint = Math.min(50, gd.heightHint);
		control.setLayoutData(gd);
		return label;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * <p>
	 * Sub-classes may override, but must call super.refresh() first.
	 */
	@Override
	public void refresh() {
		if(dbc != null) {
			dbc.dispose();
			dbc = new EMFDataBindingContext();
		}
	}

	/**
	 * Binds a positive {@link BigDecimal} value with the default number of a maximum of three
	 * fraction digits to be used in the String representation.
	 */
	protected void bindPositiveBigDecimalValue(Control text, IObservableValue<?> observedValue) {
		bindBigDecimalValue(text, observedValue, NUMBER_POSITIVE_VALIDATOR);
	}

	/**
	 * Binds a positive {@link BigDecimal} value with a user defined maximum number of fraction
	 * digits to be used in the String representation.
	 */
	protected void bindPositiveBigDecimalValue(Control text, IObservableValue<?> observedValue,
			int maxFractionDigits) {
		bindBigDecimalValue(text, observedValue, NUMBER_POSITIVE_VALIDATOR, maxFractionDigits);
	}

	/**
	 * Binds a {@link BigDecimal} value with the default number of a maximum of three fraction
	 * digits to be used in the String representation.
	 */
	private void bindBigDecimalValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator, NumberFormat nf) {

		nf.setGroupingUsed(false);
		performComplexTextBinding(dbc, text, observedValue, fromBigDecimal(nf), toBigDecimal(nf),
				FLOAT_VALIDATOR, numberValidator);
	}

	/**
	 * Binds a {@link BigDecimal} value with the default number of a maximum of three fraction
	 * digits to be used in the String representation.
	 */
	protected void bindBigDecimalValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator) {

		NumberFormat nf = getNumberInstance();
		bindBigDecimalValue(text, observedValue, numberValidator, nf);
	}

	/**
	 * Binds a BigDecimal value with a user defined maximum number of fraction digits to be
	 * used in the String representation.
	 */
	protected void bindBigDecimalValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator, int maxFractionDigits) {

		NumberFormat nf = getNumberInstance();
		nf.setMaximumFractionDigits(maxFractionDigits);
		bindBigDecimalValue(text, observedValue, numberValidator, nf);
	}

	/**
	 * Binds a positive double value with the default number of a maximum of three fraction digits
	 * to be used in the String representation.
	 */
	protected void bindPositiveDoubleValue(Control text, IObservableValue<?> observedValue) {
		bindDoubleValue(text, observedValue, NUMBER_POSITIVE_VALIDATOR);
	}

	/**
	 * Binds a positive double value with a user defined maximum number of fraction digits to be
	 * used in the String representation.
	 */
	protected void bindPositiveDoubleValue(Control text, IObservableValue<?> observedValue,
			int maxFractionDigits) {
		bindDoubleValue(text, observedValue, NUMBER_POSITIVE_VALIDATOR, maxFractionDigits);
	}

	/**
	 * Binds a double value with the default number of a maximum of three fraction digits
	 * to be used in the String representation.
	 */
	protected void bindDoubleValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator) {
		performComplexTextBinding(dbc, text, observedValue, fromDouble(false), toDouble(false),
				FLOAT_VALIDATOR, numberValidator);
	}

	/**
	 * Binds a double value with a user defined maximum number of fraction digits to be
	 * used in the String representation.
	 */
	protected void bindDoubleValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator, int maxFractionDigits) {

		NumberFormat nf = getNumberInstance();
		nf.setMaximumFractionDigits(maxFractionDigits);
		performComplexTextBinding(dbc, text, observedValue, fromDouble(nf, false),
				toDouble(nf, false), FLOAT_VALIDATOR, numberValidator);
	}

	/** Binds a positive integer value. */
	protected void bindPositiveIntegerValue(Control text, IObservableValue<?> observedValue) {
		bindIntegerValue(text, observedValue, NUMBER_POSITIVE_VALIDATOR);
	}

	/** Binds a positive or zero integer value. */
	protected void bindPositiveOrZeroIntegerValue(Control text, IObservableValue<?> observedValue) {
		bindIntegerValue(text, observedValue, NUMBER_POSITIVE_ZERO_VALIDATOR);
	}

	/** Binds a integer value. */
	protected void bindIntegerValue(Control text, IObservableValue<?> observedValue,
			IValidator<?> numberValidator) {
		performComplexTextBinding(dbc, text, observedValue, fromInteger(false), toInteger(false),
				INT_VALIDATOR, numberValidator);
	}
}
