/*-------------------------------------------------------------------------+
| Copyright 2019 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base.factory;

import java.util.List;
import java.util.Optional;

import org.fortiss.tooling.common.ui.javafx.lwfxef.controller.IController;
import org.fortiss.tooling.common.ui.javafx.lwfxef.controller.IControllerFactory;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IContentAnchorageMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IContentMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IDiagramAnchorageMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IDiagramMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.ILinkMVCBundle;
import org.fortiss.tooling.common.ui.javafx.lwfxef.mvc.IMVCBundle;

/**
 * Delegates the creational calls to concrete {@link IControllerFactory}s. The first
 * non-{@code null} element returned by the factories of the initially given list is returned. If no
 * element is created (all delegates returned {@code null}), this factory throws a
 * {@link RuntimeException}.
 * 
 * @author hoelzl
 * @author diewald
 */
public class DelegatingControllerFactory extends DelegatingFactoryBase<IControllerFactory>
		implements IControllerFactory {

	/** Constructor. */
	public DelegatingControllerFactory(List<Class<? extends IControllerFactory>> factories) {
		super(factories, null);
	}

	/** {@inheritDoc} */
	@Override
	public IController createContentController(IContentMVCBundle modelBundle) {
		Optional<IController> controller =
				getDelegateFactories().stream().map(f -> f.createContentController(modelBundle))
						.filter(cc -> cc != null).findFirst();
		if(controller.isPresent()) {
			return controller.get();
		}
		throw createNoControllerException(modelBundle);
	}

	/** {@inheritDoc} */
	@Override
	public IController createDiagramAnchorageController(IDiagramAnchorageMVCBundle modelBundle) {
		Optional<IController> controller = getDelegateFactories().stream()
				.map(f -> f.createDiagramAnchorageController(modelBundle)).filter(cc -> cc != null)
				.findFirst();
		if(controller.isPresent()) {
			return controller.get();
		}
		throw createNoControllerException(modelBundle);
	}

	/** {@inheritDoc} */
	@Override
	public IController createContentAnchorageController(IContentAnchorageMVCBundle modelBundle) {
		Optional<IController> controller = getDelegateFactories().stream()
				.map(f -> f.createContentAnchorageController(modelBundle)).filter(cc -> cc != null)
				.findFirst();
		if(controller.isPresent()) {
			return controller.get();
		}
		throw createNoControllerException(modelBundle);
	}

	/** {@inheritDoc} */
	@Override
	public IController createLinkController(ILinkMVCBundle modelBundle) {
		Optional<IController> controller = getDelegateFactories().stream()
				.map(f -> f.createLinkController(modelBundle)).filter(cc -> cc != null).findFirst();
		if(controller.isPresent()) {
			return controller.get();
		}
		throw createNoControllerException(modelBundle);
	}

	/** {@inheritDoc} */
	@Override
	public IController createDiagramController(IDiagramMVCBundle diagramBundle) {
		Optional<IController> controller = getDelegateFactories().stream().map(f -> {
			return f.createDiagramController(diagramBundle);
		}).filter(cc -> cc != null).findFirst();
		if(controller.isPresent()) {
			return controller.get();
		}
		throw createNoControllerException(diagramBundle);
	}

	/** Throws a runtime exception indicating a missing controller. */
	private RuntimeException createNoControllerException(IMVCBundle modelBundle) {
		return new RuntimeException("No controller has been specified for model elements of the" +
				" type " + modelBundle.getModel().getClass().getSimpleName() + ". Please check" +
				" the editor bindings and the delegate controllers.");
	}
}
