/*-------------------------------------------------------------------------+
| Copyright 2019 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base.factory;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.reflect.ConstructorUtils.getMatchingAccessibleConstructor;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;

import com.google.common.collect.ImmutableList;

/**
 * Base class for delegating MVC factories that wrap 1..n factories for editors of a defined type.
 * 
 * @author diewald
 */
/* package */ abstract class DelegatingFactoryBase<T> {

	/** References the edited Object in case it is needed by some factory. May be {@code null}. */
	protected final Object editedObject;

	/** References the */
	private List<T> delegateFactories = new ArrayList<>();

	/** Constructor. */
	protected DelegatingFactoryBase(List<Class<? extends T>> factories, Object editedObject) {
		this.editedObject = editedObject;

		delegateFactories = factories.stream().map(fCls -> constructFactory(fCls))
				.flatMap(Optional::stream).collect(toList());
	}

	/** Creates an instance of the given {@code delegateFactory}. */
	protected Optional<? extends T> constructFactory(Class<? extends T> delegateFactory) {
		Constructor<? extends T> ctor = null;
		try {
			ctor = getMatchingAccessibleConstructor(delegateFactory);
			try {
				return Optional.of(ctor.newInstance());
			} catch(InstantiationException | IllegalAccessException | IllegalArgumentException |
					InvocationTargetException e) {
				error(ToolingKernelUIActivator.getDefault(), "Failed to instantiate the factory " +
						delegateFactory.getSimpleName() + ".");
				return Optional.empty();
			}
		} catch(NullPointerException | SecurityException e1) {
			error(ToolingKernelUIActivator.getDefault(), "The factory " +
					delegateFactory.getSimpleName() +
					" is missing a no-arg Constructor. Only such constructors are allowed for" +
					" these factories.");
			return Optional.empty();
		}
	}

	/** Returns the list of delegate factory instances whose types are given during construction. */
	protected ImmutableList<T> getDelegateFactories() {
		return ImmutableList.copyOf(delegateFactories);
	}
}
