/*-------------------------------------------------------------------------+
| Copyright 2019 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.extension.base.factory;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.reflect.ConstructorUtils.getMatchingAccessibleConstructor;
import static org.conqat.lib.commons.collections.CollectionUtils.isNullOrEmpty;
import static org.fortiss.tooling.kernel.utils.EcoreUtils.getInterfaceType;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.eclipse.emf.ecore.EObject;
import org.fortiss.tooling.common.ui.javafx.lwfxef.model.IModelFactory;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;

/**
 * Delegates the creational and getter calls to concrete {@link IModelFactory}s. The first
 * non-{@code null} element returned by the factories of the initially given list is returned. If no
 * element is created (all delegates returned {@code null}), this factory also returns {@code null}.
 * <p>
 * Only the root model must be identical for all delegate factories. Hence, the root element of the
 * first delegate is returned.
 * 
 * @author hoelzl
 */
public class DelegatingModelFactory extends DelegatingFactoryBase<IModelFactory>
		implements IModelFactory {

	/** Constructor. */
	public DelegatingModelFactory(List<Class<? extends IModelFactory>> factories,
			Object editedObject) {
		super(factories, editedObject);
	}

	/** {@inheritDoc} */
	@Override
	protected Optional<? extends IModelFactory>
			constructFactory(Class<? extends IModelFactory> delegateFactory) {
		Constructor<? extends IModelFactory> ctor = null;
		try {
			Class<?> ctorParamType = (editedObject instanceof EObject)
					? getInterfaceType((EObject)editedObject) : editedObject.getClass();
			ctor = getMatchingAccessibleConstructor(delegateFactory, ctorParamType);
			try {
				return Optional.of(ctor.newInstance(editedObject));
			} catch(InstantiationException | IllegalAccessException | IllegalArgumentException |
					InvocationTargetException e) {
				error(ToolingKernelUIActivator.getDefault(), "Failed to instantiate the factory " +
						delegateFactory.getSimpleName() + ".");
				return empty();
			}
		} catch(NullPointerException | SecurityException e1) {
			error(ToolingKernelUIActivator.getDefault(), "The factory " +
					delegateFactory.getSimpleName() +
					" is missing a single argument Constructor accepting the edited model." +
					" Only such constructors are allowed for " +
					IModelFactory.class.getSimpleName() + "s.");
			return empty();
		}
	}

	/** {@inheritDoc} */
	@Override
	public List<?> getContentAnchorageModels(Object parent) {
		return getDelegateFactories().stream().map(f -> f.getContentAnchorageModels(parent))
				.filter(lm -> !isNullOrEmpty(lm)).flatMap(Collection::stream).distinct()
				.collect(toList());
	}

	/** {@inheritDoc} */
	@Override
	public Object getLinkStart(Object link) {
		return getDelegateFactories().stream().map(f -> f.getLinkStart(link))
				.filter(cc -> cc != null).findFirst().orElse(null);
	}

	/** {@inheritDoc} */
	@Override
	public Object getLinkEnd(Object link) {
		return getDelegateFactories().stream().map(f -> f.getLinkEnd(link)).filter(cc -> cc != null)
				.findFirst().orElse(null);
	}

	/** {@inheritDoc} */
	@Override
	public Object getParent(Object element) {
		return getDelegateFactories().stream().map(f -> f.getParent(element))
				.filter(cc -> cc != null).findFirst().orElse(null);
	}

	/** {@inheritDoc} */
	@Override
	public Object getRootModel() {
		return getDelegateFactories().stream().map(f -> f.getRootModel()).filter(cc -> cc != null)
				.findFirst().orElse(null);
	}

	/** {@inheritDoc} */
	@Override
	public List<?> getContentModels() {
		return getDelegateFactories().stream().map(f -> f.getContentModels())
				.filter(lm -> !isNullOrEmpty(lm)).flatMap(Collection::stream).distinct()
				.collect(toList());
	}

	/** {@inheritDoc} */
	@Override
	public List<?> getDiagramAnchorageModels() {
		return getDelegateFactories().stream().map(f -> f.getDiagramAnchorageModels())
				.filter(lm -> !isNullOrEmpty(lm)).flatMap(Collection::stream).distinct()
				.collect(toList());
	}

	/** {@inheritDoc} */
	@Override
	public List<?> getLinkModels() {
		return getDelegateFactories().stream().map(f -> f.getLinkModels())
				.filter(lm -> !isNullOrEmpty(lm)).flatMap(Collection::stream).distinct()
				.collect(toList());
	}

	/** {@inheritDoc} */
	@Override
	public void update() {
		// Not yet needed.
	}
}
