/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import java.util.Collection;
import java.util.EventObject;
import java.util.List;

import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.ActionFactory;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.extension.data.Prototype;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IKernelIntrospectionSystemService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.service.listener.IPersistencyServiceListener;
import org.fortiss.tooling.kernel.ui.internal.actions.CopyAction;
import org.fortiss.tooling.kernel.ui.internal.actions.CutAction;
import org.fortiss.tooling.kernel.ui.internal.actions.DeleteAction;
import org.fortiss.tooling.kernel.ui.internal.actions.PasteAction;
import org.fortiss.tooling.kernel.ui.internal.actions.RedoAction;
import org.fortiss.tooling.kernel.ui.internal.actions.RenameAction;
import org.fortiss.tooling.kernel.ui.internal.actions.SelectAllAction;
import org.fortiss.tooling.kernel.ui.internal.actions.UndoAction;
import org.fortiss.tooling.kernel.ui.internal.views.NavigatorNewMenu;
import org.fortiss.tooling.kernel.ui.service.IActionService;
import org.fortiss.tooling.kernel.ui.util.EObjectSelectionUtils;

/**
 * This class implements the {@link IActionService} interface.
 * 
 * @author hoelzlf
 */
public class ActionService implements IActionService, IPersistencyServiceListener,
		CommandStackListener, IIntrospectiveKernelService {
	/** The singleton service instance. */
	private static ActionService INSTANCE = new ActionService();

	/** Returns the service instance. */
	public static ActionService getInstance() {
		return INSTANCE;
	}

	/** The menu section ID of the global default action group. */
	public static final String GLOBAL_DEFAULT_MENU_SECTION_ID = "globalDefaults";

	/** The global delete action. */
	public DeleteAction globalDeleteAction;

	/** The global undo action. */
	public UndoAction globalUndoAction;

	/** The global redo action. */
	public RedoAction globalRedoAction;

	/** The global copy action. */
	public CopyAction globalCopyAction;

	/** The global cut action. */
	public CutAction globalCutAction;

	/** The global paste action. */
	public PasteAction globalPasteAction;

	/** The global rename action. */
	public RenameAction globalRenameAction;

	/** The global "select all" action. */
	public SelectAllAction globalSelectAllAction;

	/** Initializes the service. */
	public void initializeService() {
		globalDeleteAction = new DeleteAction();
		globalUndoAction = new UndoAction();
		globalRedoAction = new RedoAction();
		globalCopyAction = new CopyAction();
		globalCutAction = new CutAction();
		globalPasteAction = new PasteAction();
		globalRenameAction = new RenameAction();
		globalSelectAllAction = new SelectAllAction();
	}

	/** Starts the service. */
	public void startService() {
		for(ITopLevelElement context : IPersistencyService.getInstance().getTopLevelElements()) {
			topLevelElementAdded(context);
		}
		IPersistencyService.getInstance().addTopLevelElementListener(this);
		IKernelIntrospectionSystemService.getInstance().registerService(this);
	}

	/** {@inheritDoc} */
	@Override
	public void addGlobalDefaultActionSectionToMenu(IMenuManager menuManager) {
		addGlobalEditingActionsToMenu(menuManager, true, true, true, true);
	}

	/** {@inheritDoc} */
	@Override
	public void addGlobalEditingActionsToMenu(IMenuManager menuManager, boolean copyPaste,
			boolean delete, boolean rename, boolean selectAll) {
		if(copyPaste) {
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalCopyAction);
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalCutAction);
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalPasteAction);
		}
		if(delete) {
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalDeleteAction);
		}
		if(rename) {
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalRenameAction);
		}

		if(selectAll) {
			menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalSelectAllAction);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void addGlobalUndoRedoActionsToMenu(IMenuManager menuManager) {
		menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalUndoAction);
		menuManager.appendToGroup(GLOBAL_DEFAULT_MENU_SECTION_ID, globalRedoAction);
	}

	/** {@inheritDoc} */
	@Override
	public void registerGlobalActions(IActionBars actionBars) {
		registerGlobalEditingActions(actionBars);
		registerGlobalUndoRedoActions(actionBars);
		actionBars.updateActionBars();
	}

	/** Registers the global unde/redo action with the given action bar. */
	@Override
	public void registerGlobalUndoRedoActions(IActionBars actionBars) {
		registerGlobalAction(globalUndoAction, ActionFactory.UNDO.getId(),
				ActionFactory.UNDO.getCommandId(), actionBars);
		registerGlobalAction(globalRedoAction, ActionFactory.REDO.getId(),
				ActionFactory.REDO.getCommandId(), actionBars);
	}

	/** Registers the global editing action with the given action bar. */
	@Override
	public void registerGlobalEditingActions(IActionBars actionBars) {
		registerGlobalAction(globalCopyAction, ActionFactory.COPY.getId(),
				ActionFactory.COPY.getCommandId(), actionBars);
		registerGlobalAction(globalCutAction, ActionFactory.CUT.getId(),
				ActionFactory.CUT.getCommandId(), actionBars);
		registerGlobalAction(globalPasteAction, ActionFactory.PASTE.getId(),
				ActionFactory.PASTE.getCommandId(), actionBars);
		registerGlobalAction(globalDeleteAction, ActionFactory.DELETE.getId(),
				ActionFactory.DELETE.getCommandId(), actionBars);
		registerGlobalAction(globalRenameAction, ActionFactory.RENAME.getId(),
				ActionFactory.RENAME.getCommandId(), actionBars);
		registerGlobalAction(globalSelectAllAction, ActionFactory.SELECT_ALL.getId(),
				ActionFactory.SELECT_ALL.getCommandId(), actionBars);
	}

	/** Registers the given action globally. */
	private void registerGlobalAction(Action action, String actionID, String commandId,
			IActionBars actionBars) {
		action.setActionDefinitionId(commandId);
		actionBars.setGlobalActionHandler(actionID, action);
	}

	/** {@inheritDoc} */
	@Override
	public void refresh() {
		List<EObject> targets = EObjectSelectionUtils.getCurrentSelectionEObjects();
		globalCopyAction.refresh(targets);
		globalCutAction.refresh(targets);
		globalPasteAction.refresh(targets);
		globalDeleteAction.refresh(targets);
		globalRenameAction.refresh(targets);
		globalSelectAllAction.refresh(targets);
		globalUndoAction.refresh(targets);
		globalRedoAction.refresh(targets);
	}

	/** {@inheritDoc} */
	@Override
	public void commandStackChanged(EventObject event) {
		refresh();
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementLoaded(ITopLevelElement element) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementAdded(ITopLevelElement element) {
		ICommandStackService.getInstance().addCommandStackListener(element, this);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementRemoved(ITopLevelElement element) {
		ICommandStackService.getInstance().removeCommandStackListener(element, this);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementContentChanged(ITopLevelElement element) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalCopyAction() {
		globalCopyAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalPasteAction() {
		globalPasteAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalDeleteAction() {
		globalDeleteAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalUndoAction() {
		globalUndoAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalRedoAction() {
		globalRedoAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalCutAction() {
		globalCutAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalRenameAction() {
		globalRenameAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public void runGlobalSelectAllAction() {
		globalSelectAllAction.run();
	}

	/** {@inheritDoc} */
	@Override
	public Prototype getNewMenuContributionPrototype(IContributionItem contributionItem) {
		if(contributionItem instanceof ActionContributionItem) {
			ActionContributionItem aItem = (ActionContributionItem)contributionItem;
			if(aItem.getAction() instanceof NavigatorNewMenu.AddPrototypeAction) {
				NavigatorNewMenu.AddPrototypeAction action =
						(NavigatorNewMenu.AddPrototypeAction)aItem.getAction();
				return action.getPrototype();
			}
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Action Service";
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public Collection<IIntrospectionItem> getIntrospectionItems() {
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service provides access to the default actions like copy/paste, undo/redo, and delete." +
				"\nIt also handles the enabled state of these actions.";
	}
}
