/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static patched.javafx.embed.swt.SWTFXUtils.toFXImage;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.fortiss.tooling.kernel.ToolingKernelActivator;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.IKernelIntrospectionSystemService;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.extension.IContextMenuContributor;
import org.fortiss.tooling.kernel.ui.extension.IContextMenuMultiSelectionContributor;
import org.fortiss.tooling.kernel.ui.extension.data.ContextMenuContextProvider;
import org.fortiss.tooling.kernel.ui.introspection.items.ContextMenuKISSDetailsItem;
import org.fortiss.tooling.kernel.ui.service.IActionService;
import org.fortiss.tooling.kernel.ui.service.IContextMenuService;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;
import org.fortiss.tooling.kernel.ui.service.ITutorialUIService;
import org.fortiss.tooling.kernel.utils.ExtensionPointUtils;
import org.fortiss.tooling.kernel.utils.LoggingUtils;
import org.osgi.framework.Bundle;

import javafx.scene.control.MenuItem;
import javafx.scene.control.SeparatorMenuItem;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;

/**
 * This class implements the {@link IContextMenuService} interface.
 * 
 * @author hoelzl
 */
public class ContextMenuService implements IContextMenuService, IIntrospectiveKernelService {
	/** The singleton service instance. */
	private static ContextMenuService INSTANCE = new ContextMenuService();

	/** Returns the service instance. */
	public static ContextMenuService getInstance() {
		return INSTANCE;
	}

	/** The model element handler extension point ID. */
	private static final String EXTENSION_POINT_NAME =
			"org.fortiss.tooling.kernel.ui.contextMenuContribution";

	/** The model element handler configuration element name. */
	private static final String CONFIGURATION_ELEMENT_NAME = "contextMenuContribution";

	/** The model element handler class attribute name. */
	private static final String HANDLER_CLASS_ATTRIBUTE_NAME = "contributor";

	/** Stores the registered context menu contributors. */
	private final List<IContextMenuContributor> contextMenuContributorList = new ArrayList<>();

	/** An ordered list of all menu sections from top to bottom. */
	private static final String[] MENU_SECTIONS = {TOP_MOST_MENU_SECTION_ID,
			BEFORE_GLOBAL_MENU_SECTION_ID, ActionService.GLOBAL_DEFAULT_MENU_SECTION_ID,
			AFTER_GLOBAL_MENU_SECTION_ID, REPOSITORY_MENU_SECTION_ID,
			IWorkbenchActionConstants.MB_ADDITIONS, BOTTOM_MOST_MENU_SECTION_ID};

	/** Initializes the service. */
	public void initializeService() {
		// nothing to do here
	}

	/** Starts the service. */
	public void startService() {
		setupContextMenuContributors();
		IKernelIntrospectionSystemService.getInstance().registerService(this);
	}

	/** Registers the given context menu contributor with the service. */
	@Override
	public void registerContextMenuContributor(IContextMenuContributor contributor) {
		contextMenuContributorList.add(contributor);
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service allows the registration of context menu items, which are shown by right-clicking" +
				"\nin the navigator tree view. This service replaces the heavyweight mechanism of Eclipse and" +
				"\nprovides context menu items directly related to a EObject selection." +
				"\nEach context menu item can be assigned to a menu section, thus allowing a coarse ordering of items." +
				"\n\nThe service extension point is '" + EXTENSION_POINT_NAME + "'.";
	}

	/** {@inheritDoc} */
	@Override
	public MenuManager createDefaultContextMenu(ContextMenuContextProvider contextProvider) {
		MenuManager menuManager = new MenuManager("#PopupMenu");
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new MenuCreator(contextProvider));

		return menuManager;
	}

	/** Initializes the context menu contributors from plugin extensions. */
	private void setupContextMenuContributors() {
		for(IConfigurationElement ce : ExtensionPointUtils
				.getConfigurationElements(EXTENSION_POINT_NAME, CONFIGURATION_ELEMENT_NAME)) {
			Bundle bundle = ExtensionPointUtils.getBundle(ce);
			try {
				Class<?> handlerClass = ExtensionPointUtils
						.loadClass(ce.getAttribute(HANDLER_CLASS_ATTRIBUTE_NAME), bundle);

				IContextMenuContributor contextMenuContributor =
						(IContextMenuContributor)handlerClass.getConstructor().newInstance();
				contextMenuContributorList.add(contextMenuContributor);
			} catch(Exception ex) {
				LoggingUtils.error(ToolingKernelActivator.getDefault(), ex.getMessage(), ex);
			}
		}
	}

	/** Wrapper class for menu creation. */
	private class MenuCreator implements IMenuListener {

		/** Stores the context object. */
		private final ContextMenuContextProvider contextProvider;

		/** Constructor. */
		public MenuCreator(ContextMenuContextProvider contextProvider) {
			this.contextProvider = contextProvider;
		}

		/** {@inheritDoc} */
		@Override
		public void menuAboutToShow(IMenuManager menu) {
			if(!menu.isEmpty()) {
				return;
			}
			for(String section : MENU_SECTIONS) {
				addVisibleSectionSeparator(menu, section);
			}

			addContributions(menu, contextProvider);

			IActionService as = IActionService.getInstance();
			if(!ITutorialService.getInstance().isTutorialActive()) {
				as.addGlobalDefaultActionSectionToMenu(menu);
			} else {
				ITutorialUIService tsUI = ITutorialUIService.getInstance();
				as.addGlobalEditingActionsToMenu(menu, tsUI.globalCopyPasteActionsVisible(),
						tsUI.globalDeleteActionVisible(), tsUI.globalRenameActionVisible(),
						tsUI.globalSelectAllActionVisible());
			}
		}
	}

	/** Adds registered contributions to the given menu. */
	private void addContributions(IMenuManager menu, ContextMenuContextProvider contextProvider) {
		List<EObject> selection = contextProvider.getSelectedModelElementList();

		Map<String, List<IContributionItem>> contributions =
				getContributionsForElements(selection, contextProvider);

		for(String menuSectionID : contributions.keySet()) {
			List<IContributionItem> items = contributions.get(menuSectionID);
			if(items == null) {
				continue;
			}

			// populate menu
			for(IContributionItem item : items) {
				menu.appendToGroup(menuSectionID, item);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public List<MenuItem> getFXMenuItemsForElements(List<EObject> elements,
			ContextMenuContextProvider contextProvider) {
		List<MenuItem> ret = new ArrayList<MenuItem>();

		Map<String, List<IContributionItem>> contributions =
				getContributionsForElements(elements, contextProvider);

		// Create menu items for all sections
		for(String menuID : MENU_SECTIONS) {
			List<IContributionItem> menuItems = contributions.get(menuID);
			if(menuItems == null) {
				continue;
			}

			for(IContributionItem c : menuItems) {
				if(c instanceof ActionContributionItem) {
					IAction action = ((ActionContributionItem)c).getAction();

					// Create a wrapper MenuItem for the jface Action
					MenuItem menuItem = new MenuItem(action.getText());
					menuItem.setOnAction(evt -> {
						action.run();
					});

					// Add Icon
					ImageDescriptor imageDescriptor = action.getImageDescriptor();
					if(imageDescriptor != null) {
						Image fxImage =
								toFXImage(imageDescriptor.createImage().getImageData(), null);
						ImageView icon = new ImageView(fxImage);

						menuItem.setGraphic(icon);
					}

					ret.add(menuItem);
				}
			}

			// Add a separator after non-empty sections.
			if(menuItems.size() > 0) {
				ret.add(new SeparatorMenuItem());
			}
		}

		// If the last menu item is a separator, it can be removed.
		int retLastIndex = ret.size() - 1;
		if(retLastIndex > 0 && ret.get(retLastIndex) instanceof SeparatorMenuItem) {
			ret.remove(retLastIndex);
		}

		return ret;
	}

	/** {@inheritDoc} */
	@Override
	public List<MenuItem> getFXMenuItemsForElement(EObject element,
			ContextMenuContextProvider contextProvider) {
		return getFXMenuItemsForElements(asList(element), contextProvider);
	}

	/**
	 * Retrieves a map from menu section ids to a list {@link IContributionItem}s for the given
	 * selectedElements.
	 */
	private Map<String, List<IContributionItem>> getContributionsForElements(
			List<EObject> selectedElements, ContextMenuContextProvider contextProvider) {
		Map<String, List<IContributionItem>> ret = new HashMap<String, List<IContributionItem>>();

		for(IContextMenuContributor contributor : contextMenuContributorList) {
			if(!INavigatorService.getInstance().isExpertViewActive() &&
					contributor.hiddenInNonExpertView()) {
				continue;
			}

			// get contributions from contributor
			List<IContributionItem> items = null;
			boolean isTutorialActive = ITutorialService.getInstance().isTutorialActive();
			if(selectedElements.size() == 1) {
				// Size checked.
				EObject selectedElement = selectedElements.get(0);
				items = contributor.getContributedItems(selectedElement, contextProvider);

				// active tutorial may filter contributions
				if(isTutorialActive) {
					ITutorialUIService service = ITutorialUIService.getInstance();
					Predicate<? super IContributionItem> isVisible =
							i -> service.contextMenuContributionVisible(selectedElement, i);
					items = items.stream().filter(isVisible).collect(Collectors.toList());
				}
			} else if(selectedElements.size() > 1 &&
					contributor instanceof IContextMenuMultiSelectionContributor) {
				// active tutorial does not support multiple selections
				if(isTutorialActive) {
					continue;
				}

				IContextMenuMultiSelectionContributor multiContributor =
						(IContextMenuMultiSelectionContributor)contributor;
				items = multiContributor.getContributedItems(selectedElements, contextProvider);
			}

			if(items == null) {
				continue;
			}

			String menuSectionID = contributor.getMenuSectionID();
			if(menuSectionID == null) {
				menuSectionID = IWorkbenchActionConstants.MB_ADDITIONS;
			}

			List<IContributionItem> sectionItems = ret.get(menuSectionID);
			if(sectionItems == null) {
				sectionItems = new ArrayList<IContributionItem>();
				ret.put(menuSectionID, sectionItems);
			}

			sectionItems.addAll(items);
		}

		return ret;
	}

	/** Adds a visible section separator. */
	private void addVisibleSectionSeparator(IMenuManager menu, String sectionName) {
		Separator separator = new Separator(sectionName);
		separator.setVisible(true);
		menu.add(separator);
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Context Menu Service";
	}

	/** {@inheritDoc} */
	@Override
	public List<IIntrospectionItem> getIntrospectionItems() {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return new ContextMenuKISSDetailsItem(contextMenuContributorList);
	}
}
