/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static java.util.Collections.emptyList;
import static java.util.Collections.sort;
import static org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator.getFXImage;
import static org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator.getImageDescriptor;
import static org.fortiss.tooling.kernel.utils.EcoreUtils.postRefreshNotification;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.progress.UIJob;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.service.IConstraintCheckerService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.service.listener.IPersistencyServiceListener;
import org.fortiss.tooling.kernel.ui.internal.views.marker.MarkerViewController;
import org.fortiss.tooling.kernel.ui.service.IMarkerService;

import javafx.scene.image.ImageView;

/**
 * This class implements the {@link IMarkerService} interface.
 * 
 * @author hoelzl
 */
public class MarkerService implements IMarkerService, IPersistencyServiceListener {
	/** The singleton service instance. */
	private static MarkerService INSTANCE = new MarkerService();

	/** Cache for shared copies of small version of images. */
	private Map<ESeverity, Image> smallSeverityImageMap = new HashMap<ESeverity, Image>();

	/** Cache for shared copies of large version of images. */
	private Map<ESeverity, Image> largeSeverityImageMap = new HashMap<ESeverity, Image>();

	/** Returns the service instance. */
	public static MarkerService getInstance() {
		return INSTANCE;
	}

	/** ID of this decorator. */
	public static final String ID = "org.fortiss.tooling.kernel.ui.internal.MarkerService";

	/** Stores the violations for each {@link ITopLevelElement}. */
	private final HashMap<ITopLevelElement, CacheEntry> violationCache =
			new HashMap<ITopLevelElement, CacheEntry>();

	/** Stores the UI update job. */
	private final UIJob updateUI = new UIJob("Update Model Decorators") {
		@Override
		public IStatus runInUIThread(IProgressMonitor monitor) {
			if(markerController != null) {
				markerController.refresh();
			}
			return Status.OK_STATUS;
		}
	};

	/** Invalid top-level elements to be refreshed by the constraint checker job. */
	private final Queue<ITopLevelElement> invalidElements =
			new ConcurrentLinkedQueue<ITopLevelElement>();

	/** Stores the marker view. */
	private MarkerViewController markerController;

	/** Stores the constraint checking job. */
	private final Job constraintCheckerJob = new Job("Model Constraint Checker Job") {

		@Override
		protected IStatus run(IProgressMonitor monitor) {

			do {
				if(monitor.isCanceled()) {
					return Status.CANCEL_STATUS;
				}

				ITopLevelElement toBeRefreshed = invalidElements.poll();
				if(toBeRefreshed == null) {
					return Status.OK_STATUS;
				}

				refreshMarkers(toBeRefreshed);
			} while(true);
		}
	};

	/** Initializes the service. */
	public void initializeService() {
		// nothing to do here
	}

	/** Starts the service. */
	public void startService() {
		for(ITopLevelElement element : IPersistencyService.getInstance().getTopLevelElements()) {
			invalidElements.add(element);
		}
		constraintCheckerJob.schedule();
		IPersistencyService.getInstance().addTopLevelElementListener(this);
	}

	/** Sets the current marker view. */
	public void setMarkerViewPart(MarkerViewController markerController) {
		this.markerController = markerController;
	}

	/** {@inheritDoc} */
	@Override
	public Collection<IConstraintViolation<? extends EObject>> getViolations(EObject element) {
		ITopLevelElement top = IPersistencyService.getInstance().getTopLevelElementFor(element);
		if(top == null) {
			return Collections.emptyList();
		}
		CacheEntry ce = getCacheEntry(top);
		Collection<IConstraintViolation<? extends EObject>> coll = ce.getCachedList(element);
		if(coll.isEmpty()) {
			return ce.getChildCachedList(element);
		}
		return coll;
	}

	/** {@inheritDoc} */
	@Override
	public ESeverity getHighestViolationSeverity(EObject element) {
		ITopLevelElement top = IPersistencyService.getInstance().getTopLevelElementFor(element);
		if(top == null) {
			return ESeverity.lowest();
		}
		ESeverity sev = getCacheEntry(top).getHighestSeverity(element);
		return sev;
	}

	/** Accesses cache. */
	private CacheEntry getCacheEntry(ITopLevelElement element) {
		synchronized(violationCache) {
			CacheEntry result = violationCache.get(element);
			if(result == null) {
				result = new CacheEntry(element);
				violationCache.put(element, result);
			}
			return result;
		}
	}

	/** {@inheritDoc} */
	@Override
	public void refreshMarkers(ITopLevelElement element) {

		EObject rootModelElement = element.getRootModelElement();
		IConstraintCheckerService ccs = IConstraintCheckerService.getInstance();
		List<IConstraintViolation<? extends EObject>> checkResult =
				ccs.performAllConstraintChecksRecursively(rootModelElement);

		ccs.performAllAsynchronousConstraintChecksRecursively(rootModelElement, violations -> {
			synchronized(violationCache) {
				CacheEntry entry = getCacheEntry(element);
				entry.addNewViolations(violations);
			}

			updateUI.schedule();
		});

		synchronized(violationCache) {
			CacheEntry entry = getCacheEntry(element);
			entry.updateCacheEntries(checkResult);
		}

		updateUI.schedule();
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementLoaded(ITopLevelElement element) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementAdded(ITopLevelElement element) {
		doConstraintCheck(element);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementContentChanged(ITopLevelElement element) {
		doConstraintCheck(element);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementRemoved(ITopLevelElement element) {
		clearViolationCache(element);
	}

	/** Schedules the given element for constraint checking. */
	private void doConstraintCheck(ITopLevelElement element) {
		invalidElements.add(element);
		constraintCheckerJob.schedule();
	}

	/**
	 * Clears the cache from all elements associated with the given top-level
	 * element.
	 */
	private void clearViolationCache(ITopLevelElement element) {
		synchronized(violationCache) {
			violationCache.remove(element);
			updateUI.schedule();
		}
	}

	/** Marker service cache entry. */
	private static class CacheEntry {

		/** Stores the mapping from model elements to violation lists. */
		private final Map<EObject, List<IConstraintViolation<? extends EObject>>> violationsMap =
				new ConcurrentHashMap<EObject, List<IConstraintViolation<? extends EObject>>>();

		/** Stores the highest severity value for each cached element. */
		private final Map<EObject, ESeverity> highestSeverityMap =
				new ConcurrentHashMap<EObject, ESeverity>();

		/** Stores the child causing the highest severity value for each cached element. */
		private final Map<EObject, EObject> highestSeverityChildMap =
				new ConcurrentHashMap<EObject, EObject>();

		/** Stores the top-level element. */
		private ITopLevelElement topElement;

		/** Stores the display update UI job. */
		private final UIJob refreshElementDisplay = new UIJob("Update Markers") {

			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {
				for(EObject eo : violationsMap.keySet()) {
					postRefreshNotification(eo);
					// Create copy of list since it might be updated in parallel to the UI update.
					for(IConstraintViolation<? extends EObject> viol : new ArrayList<>(
							getCachedList(eo))) {
						postRefreshNotification(viol.getSource());
					}
				}
				return Status.OK_STATUS;
			}
		};

		/** Constructor. */
		public CacheEntry(ITopLevelElement topElement) {
			this.topElement = topElement;
		}

		/** Returns the highest severity for the given element. */
		public ESeverity getHighestSeverity(EObject element) {
			ESeverity sev = highestSeverityMap.get(element);
			return sev == null ? ESeverity.lowest() : sev;
		}

		/** Updates the cache entry. */
		public synchronized void
				updateCacheEntries(List<IConstraintViolation<? extends EObject>> newViolations) {
			clearCachedLists();
			// update cache entries
			addNewViolations(newViolations);
		}

		/** Add the given violations to the cache entry. */
		public synchronized void
				addNewViolations(List<IConstraintViolation<? extends EObject>> newViolations) {
			for(IConstraintViolation<? extends EObject> violation : newViolations) {
				getCachedList(violation.getSource()).add(violation);
			}
			// fix cached list order
			for(EObject eo : violationsMap.keySet()) {
				List<IConstraintViolation<? extends EObject>> list = violationsMap.get(eo);
				if(list.isEmpty()) {
					highestSeverityMap.put(eo, ESeverity.lowest());
					highestSeverityChildMap.remove(eo);
				} else {
					sort(list, IConstraintViolation.SEVERITY_COMPARATOR);
					// list is guaranteed to be non-empty by above check => get(0) is fine
					IConstraintViolation<? extends EObject> violation = list.get(0);
					highestSeverityMap.put(eo, violation.getSeverity());
					highestSeverityChildMap.put(eo, violation.getSource());
				}
			}
			// project severity to parent
			computeHighestSeverity(topElement.getRootModelElement());
			// refresh display
			refreshElementDisplay.schedule();
		}

		/** Recursively projects highest severity from children to parents. */
		private ESeverity computeHighestSeverity(EObject element) {
			ESeverity severity = getHighestSeverity(element);
			EObject causingChild = highestSeverityChildMap.get(element);
			for(EObject child : element.eContents()) {
				ESeverity childSeverity = computeHighestSeverity(child);
				if(ESeverity.getIntSeverity(severity) > ESeverity.getIntSeverity(childSeverity)) {
					severity = childSeverity;
					causingChild = child;
				}
			}
			highestSeverityMap.put(element, severity);
			if(causingChild != null) {
				highestSeverityChildMap.put(element, causingChild);
			}
			return severity;
		}

		/** Returns the cached list instance creating it if necessary. */
		private List<IConstraintViolation<? extends EObject>> getCachedList(EObject element) {
			List<IConstraintViolation<? extends EObject>> list = violationsMap.get(element);
			if(list == null) {
				list = new ArrayList<IConstraintViolation<? extends EObject>>();
				violationsMap.put(element, list);
			}
			return list;
		}

		/** Returns the cached list of child causing the violation of the given element. */
		public List<IConstraintViolation<? extends EObject>> getChildCachedList(EObject element) {
			if(!highestSeverityChildMap.containsKey(element)) {
				return emptyList();
			}
			return getCachedList(highestSeverityChildMap.get(element));
		}

		/** Returns all violations of the given severity. */
		public List<IConstraintViolation<? extends EObject>>
				getAllViolationsWithSeverity(ESeverity severity) {
			List<IConstraintViolation<? extends EObject>> list =
					new ArrayList<IConstraintViolation<? extends EObject>>();
			for(EObject element : violationsMap.keySet()) {
				for(IConstraintViolation<? extends EObject> violation : violationsMap
						.get(element)) {
					if(violation.getSeverity() == severity) {
						list.add(violation);
					}
				}
			}
			return list;
		}

		/** Clears all cached lists. */
		private void clearCachedLists() {
			for(List<IConstraintViolation<? extends EObject>> list : violationsMap.values()) {
				list.clear();
			}
			for(EObject eo : highestSeverityMap.keySet()) {
				highestSeverityMap.put(eo, ESeverity.lowest());
				highestSeverityChildMap.remove(eo);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public Collection<IConstraintViolation<? extends EObject>>
			getAllViolationsWithSeverity(ESeverity severity) {
		List<IConstraintViolation<? extends EObject>> list =
				new ArrayList<IConstraintViolation<? extends EObject>>();
		for(ITopLevelElement top : IPersistencyService.getInstance().getTopLevelElements()) {
			list.addAll(getCacheEntry(top).getAllViolationsWithSeverity(severity));
		}
		return list;
	}

	/** {@inheritDoc} */
	@Override
	public ImageView getFXImageFor(ESeverity severity, boolean smallDecoration) {
		switch(severity) {
			case FATAL:
				return getFXImage("icons/fatal.gif");
			case ERROR:
				if(smallDecoration) {
					return getFXImage("icons/error_small.gif");
				}
				return getFXImage("icons/error.gif");
			case WARNING:
				if(smallDecoration) {
					return getFXImage("icons/warning_small.gif");
				}
				return getFXImage("icons/warning.gif");
			case INFO:
				return getFXImage("icons/info.gif");
			case DEBUG:
				return getFXImage("icons/debug.gif");
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public ImageDescriptor getImageFor(ESeverity severity, boolean smallDecoration) {
		switch(severity) {
			case FATAL:
				return getImageDescriptor("icons/fatal.gif");
			case ERROR:
				if(smallDecoration) {
					return getImageDescriptor("icons/error_small.gif");
				}
				return getImageDescriptor("icons/error.gif");
			case WARNING:
				if(smallDecoration) {
					return getImageDescriptor("icons/warning_small.gif");
				}
				return getImageDescriptor("icons/warning.gif");
			case INFO:
				return getImageDescriptor("icons/info.gif");
			case DEBUG:
				return getImageDescriptor("icons/debug.gif");
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public Image getSharedImageFor(ESeverity severity, boolean smallDecoration) {
		if(smallDecoration) {
			return getImage(severity, smallDecoration, smallSeverityImageMap);
		}
		return getImage(severity, smallDecoration, largeSeverityImageMap);
	}

	/** Helper for {@link #getSharedImageFor(ESeverity, boolean)}. */
	private Image getImage(ESeverity severity, boolean smallDecoration,
			Map<ESeverity, Image> imageMap) {
		Image image = imageMap.get(severity);
		if(image == null) {
			ImageDescriptor imgDescr = getImageFor(severity, smallDecoration);
			if(imgDescr != null) {
				image = imgDescr.createImage();
				imageMap.put(severity, image);
			}
		}
		return image;
	}
}
