/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static java.util.Collections.emptyList;
import static java.util.Collections.sort;
import static org.fortiss.tooling.kernel.utils.KernelModelElementUtils.isChildElementOf;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.EventObject;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPartService;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.fortiss.tooling.kernel.ToolingKernelActivator;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IKernelIntrospectionSystemService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.service.base.EObjectAwareServiceBase;
import org.fortiss.tooling.kernel.ui.extension.IModelEditor;
import org.fortiss.tooling.kernel.ui.extension.IModelEditorBinding;
import org.fortiss.tooling.kernel.ui.extension.ModelEditorNotAvailableBinding;
import org.fortiss.tooling.kernel.ui.internal.editor.ExtendableMultiPageEditor;
import org.fortiss.tooling.kernel.ui.internal.editor.ModelElementEditorInput;
import org.fortiss.tooling.kernel.ui.introspection.items.ModelEditorBindingKISSDetailsItem;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;

/**
 * This class implements the {@link IModelEditorBindingService} interface.
 * 
 * @author hoelzl
 */
public class ModelEditorBindingService extends EObjectAwareServiceBase<IModelEditorBinding<EObject>>
		implements IModelEditorBindingService, IPartListener, CommandStackListener,
		IIntrospectiveKernelService {
	/** The singleton service instance. */
	private static ModelEditorBindingService INSTANCE = new ModelEditorBindingService();

	/** Returns the service instance. */
	public static ModelEditorBindingService getInstance() {
		return INSTANCE;
	}

	/** The compositor extension point ID. */
	private static final String EXTENSION_POINT_NAME =
			"org.fortiss.tooling.kernel.ui.modelEditorBinding";

	/** The compositor configuration element name. */
	private static final String CONFIGURATION_ELEMENT_NAME = "modelEditorBinding";

	/** The handler class attribute name. */
	private static final String HANDLER_CLASS_ATTRIBUTE_NAME = "binding";

	/** Stores the currently opened editors and their model elements. */
	private final Map<IEditorPart, EObject> currentEditors = new HashMap<IEditorPart, EObject>();

	/** Stores the command stack registrations. */
	private final Map<EObject, ITopLevelElement> currentCommandStacks =
			new HashMap<EObject, ITopLevelElement>();

	/** {@inheritDoc} */
	@Override
	public void startService() {
		IKernelIntrospectionSystemService.getInstance().registerService(this);

		for(List<IModelEditorBinding<EObject>> bindings : handlerMap.values()) {
			sort(bindings, new Comparator<IModelEditorBinding<EObject>>() {
				@Override
				public int compare(IModelEditorBinding<EObject> o1,
						IModelEditorBinding<EObject> o2) {
					return o2.getPriority() - o1.getPriority();
				}
			});
		}
	}

	/** Registers the given editor binding with the service. */
	@Override
	@SuppressWarnings("unchecked")
	public <T extends EObject> void registerEditorBinding(IModelEditorBinding<T> binding,
			Class<T> modelElementClass) {
		addHandler(modelElementClass, (IModelEditorBinding<EObject>)binding);
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service provides the registration mechanism for editors of model elements." +
				"\nAll model editors for a given model element are displayed in a tabbed multi-page editor." +
				"\n\nThe service extension point is '" + EXTENSION_POINT_NAME + "'.";
	}

	/** {@inheritDoc} */
	@Override
	public void openInEditor(EObject element) {
		IEditorPart part = openEditor(element);

		// The opened editor may not be an editor for the element itself but for a container of it.
		// In this case, navigate to the element in the editor.
		if(part instanceof IModelEditor<?>) {
			IModelEditor<?> editor = (IModelEditor<?>)part;
			if(editor.getEditedObject() != element) {
				editor.navigateTo(element);
			}
		}
	}

	/** Opens an editor for the given element that shows the containedElement */
	private IEditorPart openEditor(EObject element) {
		List<IModelEditorBinding<EObject>> bindings = getBindings(element);
		int numBindings = bindings.size();

		// Recurse if there is no binding or no editor for the current model element.
		boolean noEditor =
				numBindings == 1 && bindings.get(0) instanceof ModelEditorNotAvailableBinding;
		boolean noBindings = numBindings == 0;
		if(noBindings || noEditor) {
			if(element.eContainer() != null) {
				return openEditor(element.eContainer());
			}
			return null;
		}

		try {
			IWorkbench workbench = PlatformUI.getWorkbench();
			IWorkbenchPage activePage = workbench.getActiveWorkbenchWindow().getActivePage();
			IEditorPart part = activePage.openEditor(new ModelElementEditorInput(element),
					ExtendableMultiPageEditor.ID);
			currentEditors.put(part, element);
			// ensure registration with part service
			IPartService service = part.getSite().getService(IPartService.class);
			service.addPartListener(new EditorSitePartListener());
			// ensure registration with command stack service
			ITopLevelElement top = IPersistencyService.getInstance().getTopLevelElementFor(element);
			ICommandStackService.getInstance().addCommandStackListener(top, this);
			currentCommandStacks.put(element, top);
			return part;
		} catch(final PartInitException e) {
			error(ToolingKernelActivator.getDefault(),
					"Could not open editor with ID " + ExtendableMultiPageEditor.ID, e);
			return null;
		}
	}

	/** {@inheritDoc} */
	@Override
	public List<IModelEditorBinding<EObject>> getBindings(Class<? extends EObject> elementType) {
		List<IModelEditorBinding<EObject>> bindings = getRegisteredHandlers(elementType);
		if(bindings == null) {
			bindings = emptyList();
		}
		return new ArrayList<>(bindings);
	}

	/** {@inheritDoc} */
	@Override
	public List<IModelEditorBinding<EObject>> getBindings(EObject element) {
		return getBindings(element.getClass());
	}

	/** {@inheritDoc} */
	@Override
	protected String getExtensionPointName() {
		return EXTENSION_POINT_NAME;
	}

	/** {@inheritDoc} */
	@Override
	protected String getConfigurationElementName() {
		return CONFIGURATION_ELEMENT_NAME;
	}

	/** {@inheritDoc} */
	@Override
	protected String getHandlerClassAttribute() {
		return HANDLER_CLASS_ATTRIBUTE_NAME;
	}

	/** {@inheritDoc} */
	@Override
	public void closeEditors(EObject parentElement) {
		IEditorPart[] parts = new IEditorPart[0];
		for(IEditorPart editor : currentEditors.keySet().toArray(parts)) {
			EObject editedElement = currentEditors.get(editor);
			if(isChildElementOf(editedElement, parentElement)) {
				closeEditor(editor);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public void closeEditor(EObject element) {
		IEditorPart[] parts = new IEditorPart[0];
		for(IEditorPart editor : currentEditors.keySet().toArray(parts)) {
			EObject editedElement = currentEditors.get(editor);
			if(editedElement.equals(element)) {
				closeEditor(editor);
			}
		}
	}

	/** Closes the given editor. */
	private void closeEditor(final IEditorPart editor) {
		IWorkbenchPartSite editorSite = editor.getSite();
		IWorkbenchPage page = editorSite.getPage();
		editorSite.getShell().getDisplay().asyncExec(() -> page.closeEditor(editor, false));
	}

	/** {@inheritDoc} */
	@Override
	public void partActivated(IWorkbenchPart part) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void partBroughtToTop(IWorkbenchPart part) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void partClosed(IWorkbenchPart part) {
		if(currentEditors.containsKey(part)) {
			EObject eo = currentEditors.get(part);
			ITopLevelElement top = currentCommandStacks.get(eo);
			currentCommandStacks.remove(eo);
			currentEditors.remove(part);

			// clean up command stack registration
			for(IEditorPart other : currentEditors.keySet()) {
				if(top == currentCommandStacks.get(currentEditors.get(other))) {
					return;
				}
			}
			// no other editor references the command stack of top anymore
			ICommandStackService.getInstance().removeCommandStackListener(top, this);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void partDeactivated(IWorkbenchPart part) {
		// ignore
	}

	/** {@inheritDoc} */
	@Override
	public void partOpened(IWorkbenchPart part) {
		// can be ignored
	}

	/** {@inheritDoc} */
	@Override
	public void commandStackChanged(EventObject event) {
		IEditorPart[] parts = new IEditorPart[0];
		for(IEditorPart part : currentEditors.keySet().toArray(parts)) {
			IPersistencyService ps = IPersistencyService.getInstance();
			ITopLevelElement top = ps.getTopLevelElementFor(currentEditors.get(part));
			if(top == null) {
				closeEditor(part);
			}
		}
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	public IModelEditor<EObject> getActiveEditor() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IEditorPart activeEditor =
				workbench.getActiveWorkbenchWindow().getActivePage().getActiveEditor();
		if(activeEditor instanceof ExtendableMultiPageEditor) {
			ExtendableMultiPageEditor extEditor = (ExtendableMultiPageEditor)activeEditor;
			IModelEditor<EObject> inner = extEditor.getActiveModelEditor();
			if(inner != null) {
				return inner;
			}
			return extEditor;
		}
		if(activeEditor instanceof IModelEditor) {
			return (IModelEditor<EObject>)activeEditor;
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isOpen(EObject element) {
		return currentEditors.containsValue(element);
	}

	/**
	 * Instance class used for listening to editor part life-cycle. This class is needed, because
	 * Eclipse removes this instance once the editor site is disposed. Therefore, we cannot use the
	 * service singleton directly, since multiple editors would add it, but the first closed one
	 * removes it.
	 */
	private class EditorSitePartListener implements IPartListener {

		/** {@inheritDoc} */
		@Override
		public void partActivated(IWorkbenchPart part) {
			ModelEditorBindingService.this.partActivated(part);
		}

		/** {@inheritDoc} */
		@Override
		public void partBroughtToTop(IWorkbenchPart part) {
			ModelEditorBindingService.this.partBroughtToTop(part);
		}

		/** {@inheritDoc} */
		@Override
		public void partClosed(IWorkbenchPart part) {
			ModelEditorBindingService.this.partClosed(part);
		}

		/** {@inheritDoc} */
		@Override
		public void partDeactivated(IWorkbenchPart part) {
			ModelEditorBindingService.this.partDeactivated(part);
		}

		/** {@inheritDoc} */
		@Override
		public void partOpened(IWorkbenchPart part) {
			ModelEditorBindingService.this.partOpened(part);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void closeAllEditors() {
		IEditorPart[] parts = new IEditorPart[0];
		for(IEditorPart editor : currentEditors.keySet().toArray(parts)) {
			closeEditor(editor);
		}
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Model Editor Binding Service";
	}

	/** {@inheritDoc} */
	@Override
	public List<IIntrospectionItem> getIntrospectionItems() {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return new ModelEditorBindingKISSDetailsItem(handlerMap);
	}
}
