/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static java.util.Collections.emptyList;
import static org.conqat.lib.commons.string.StringUtils.wrapLongLines;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;

import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.swt.graphics.Image;
import org.fortiss.tooling.kernel.ToolingKernelActivator;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.IKernelIntrospectionSystemService;
import org.fortiss.tooling.kernel.service.base.EObjectAwareServiceBase;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.introspection.items.ModelElementHandlerKISSDetailsItem;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;

import javafx.scene.Node;

/**
 * This class implements the {@link IModelElementHandlerService} interface. It
 * requires a unique and singleton model element handler per model element
 * class.
 * 
 * @author hoelzl
 */
public class ModelElementHandlerService
		extends EObjectAwareServiceBase<IModelElementHandler<EObject>>
		implements IModelElementHandlerService, IIntrospectiveKernelService {
	/** The singleton service instance. */
	private static ModelElementHandlerService INSTANCE = new ModelElementHandlerService();

	/** Returns the service instance. */
	public static ModelElementHandlerService getInstance() {
		return INSTANCE;
	}

	/** The model element handler extension point ID. */
	private static final String EXTENSION_POINT_NAME =
			"org.fortiss.tooling.kernel.ui.modelElementHandler";

	/** The model element handler configuration element name. */
	private static final String CONFIGURATION_ELEMENT_NAME = "modelElementHandler";

	/** The model element handler class attribute name. */
	private static final String HANDLER_CLASS_ATTRIBUTE_NAME = "handler";

	/** {@inheritDoc} */
	@Override
	public void startService() {
		IKernelIntrospectionSystemService.getInstance().registerService(this);
	}

	/** Registers the given handler with the service. */
	@Override
	@SuppressWarnings("unchecked")
	public <T extends EObject> void registerModelElementHandler(IModelElementHandler<T> handler,
			Class<T> modelElementClass) {
		addHandler(modelElementClass, (IModelElementHandler<EObject>)handler);
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service provides the user interface wrappers for model elements." +
				"\nThe navigator service uses the handlers to build the hierarchiy model structure" +
				"\nas well as the display labels and icons. The model element handlers are the " +
				"\ncentral point for transforming the content of the model into UI information." +
				"\n\nThe service extension point is '" + EXTENSION_POINT_NAME + "'.";
	}

	/** {@inheritDoc} */
	@Override
	public IModelElementHandler<EObject> getModelElementHandler(EObject modelElement) {
		Class<?> clazz = modelElement.getClass();
		List<IModelElementHandler<EObject>> handlerList = getRegisteredHandlers(clazz);
		if(handlerList == null || handlerList.isEmpty()) {
			error(ToolingKernelActivator.getDefault(), "ERROR: " +
					modelElement.getClass().getName() + " does not have a registered handler!");
			return null;
		}
		// Handler list guaranteed to be non-empty.
		return handlerList.get(0);
	}

	/** {@inheritDoc} */
	@Override
	protected boolean requiresUniqueHandler() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	protected String getExtensionPointName() {
		return EXTENSION_POINT_NAME;
	}

	/** {@inheritDoc} */
	@Override
	protected String getConfigurationElementName() {
		return CONFIGURATION_ELEMENT_NAME;
	}

	/** {@inheritDoc} */
	@Override
	protected String getHandlerClassAttribute() {
		return HANDLER_CLASS_ATTRIBUTE_NAME;
	}

	/** {@inheritDoc} */
	@Override
	public String getName(EObject modelElement) {
		IModelElementHandler<EObject> handler = getModelElementHandler(modelElement);
		return handler == null ? null : handler.getName(modelElement);
	}

	/** {@inheritDoc} */
	@Override
	public String getDescription(EObject modelElement) {
		IModelElementHandler<EObject> handler = getModelElementHandler(modelElement);
		return handler == null ? null : handler.getDescription(modelElement);
	}

	/** {@inheritDoc} */
	@Override
	public Image getIcon(EObject modelElement) {
		IModelElementHandler<EObject> handler = getModelElementHandler(modelElement);
		return handler == null ? null : handler.getIcon(modelElement);
	}

	/** {@inheritDoc} */
	@Override
	public Node getFXIcon(EObject modelElement) {
		IModelElementHandler<EObject> handler = getModelElementHandler(modelElement);
		return handler == null ? null : handler.getFXIcon(modelElement);
	}

	/** {@inheritDoc} */
	@Override
	public String getWrappedNameDescription(EObject modelElement) {
		IModelElementHandler<EObject> handler = getModelElementHandler(modelElement);
		if(handler == null) {
			return "";
		}
		return wrapLongLines(
				handler.getName(modelElement) + ": " + handler.getDescription(modelElement), 100);
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Model Element Handler Service";
	}

	/** {@inheritDoc} */
	@Override
	public List<IIntrospectionItem> getIntrospectionItems() {
		return emptyList();
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return new ModelElementHandlerKISSDetailsItem(handlerMap);
	}
}
