/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static org.conqat.ide.commons.ui.logging.LoggingUtils.error;

import java.io.IOException;
import java.util.Collection;
import java.util.EventObject;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.ISaveablesLifecycleListener;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.Saveable;
import org.eclipse.ui.SaveablesLifecycleEvent;
import org.eclipse.ui.progress.UIJob;
import org.fortiss.tooling.kernel.ToolingKernelActivator;
import org.fortiss.tooling.kernel.extension.ITutorialProvider;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.extension.data.TutorialAtomicStep;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.service.listener.IPersistencyServiceListener;
import org.fortiss.tooling.kernel.service.listener.ITutorialServiceListener;
import org.fortiss.tooling.kernel.ui.extension.data.ContextMenuContextProvider;
import org.fortiss.tooling.kernel.ui.internal.views.NavigatorViewPart;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;

/**
 * This class implements the {@link INavigatorService} interface.
 * 
 * @author hoelzl
 */
public class NavigatorService implements INavigatorService, IPersistencyServiceListener,
		CommandStackListener, ITutorialServiceListener, IIntrospectiveKernelService {
	/** The singleton service instance. */
	private static NavigatorService INSTANCE = new NavigatorService();

	/** Returns the service instance. */
	public static NavigatorService getInstance() {
		return INSTANCE;
	}

	/** Stores the navigator view. */
	private NavigatorViewPart navigatorViewPart;

	/** Stores whether the expert view is active. **/
	private boolean expertViewActive = false;

	/** Stores the {@link Saveable}s for the top-level elements. */
	private Map<ITopLevelElement, TopLevelElementSaveable> saveables =
			new HashMap<ITopLevelElement, NavigatorService.TopLevelElementSaveable>();

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Navigator Service";
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service manages the model element navigator view including the handling of" +
				"\nsaveables, the dirty state, and the expert mode.";
	}

	/** Initializes the service. */
	public void initializeService() {
		IPersistencyService.getInstance().addTopLevelElementListener(this);
	}

	/** Starts the service. */
	public void startService() {
		for(ITopLevelElement element : IPersistencyService.getInstance().getTopLevelElements()) {
			ICommandStackService.getInstance().addCommandStackListener(element, this);
			saveables.put(element, new TopLevelElementSaveable(element));
		}
		ITutorialService.getInstance().addTutorialServiceListener(this);
	}

	/** Sets the navigator view. */
	public void setNavigatorViewPart(NavigatorViewPart part) {
		this.navigatorViewPart = part;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isNavigatorView(ContextMenuContextProvider provider) {
		return provider == navigatorViewPart;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isExpertViewActive() {
		return expertViewActive;
	}

	/** Toggles the expert mode. */
	public void toggleExpertMode() {
		expertViewActive = !expertViewActive;
	}

	/** Returns the navigator view part. */
	public NavigatorViewPart getNavigatorViewPart() {
		return navigatorViewPart;
	}

	/** Returns the {@link #navigatorViewPart}'s {@link ISelectionProvider}. */
	private ISelectionProvider getSelectionProvider() {
		return navigatorViewPart.getSite().getSelectionProvider();
	}

	/** {@inheritDoc} */
	@Override
	public ISelection getCurrentSelection() {
		return getSelectionProvider().getSelection();
	}

	/** {@inheritDoc} */
	@Override
	public void setCurrentSelection(ISelection selection) {
		getSelectionProvider().setSelection(selection);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementLoaded(ITopLevelElement element) {
		addTopLevelElementScheduled(element);
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementAdded(final ITopLevelElement element) {
		addTopLevelElementScheduled(element);
	}

	/** Adds the given {@link ITopLevelElement} to the Navigator and updates the view. */
	private void addTopLevelElementScheduled(final ITopLevelElement element) {
		new UIJob("NavigatorServiceSafeableRefresh") {
			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {
				element.addCommandStackListener(NavigatorService.this);
				saveables.put(element, new TopLevelElementSaveable(element));
				notifySaveablesPostOpen(element);
				refresh();
				return Status.OK_STATUS;
			}
		}.schedule();
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementRemoved(final ITopLevelElement element) {
		if(element == null) {
			return;
		}

		new UIJob("NavigatorServiceSafeableRefresh") {
			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {

				notifySaveablesPreClose(element);
				element.removeCommandStackListener(NavigatorService.this);
				saveables.remove(element);
				notifySaveablesPostClose(element);
				refresh();
				return Status.OK_STATUS;
			}
		}.schedule();
	}

	/** {@inheritDoc} */
	@Override
	public void topLevelElementContentChanged(final ITopLevelElement element) {
		new UIJob("NavigatorServiceSafeableRefresh") {
			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {
				notifySaveablesDirtyChanged(element);
				refresh();
				return Status.OK_STATUS;
			}
		}.schedule();
	}

	/** {@inheritDoc} */
	@Override
	public void commandStackChanged(EventObject event) {
		if(event.getSource() instanceof ITopLevelElement) {
			notifySaveablesDirtyChanged((ITopLevelElement)event.getSource());
		}
		refresh();
	}

	/** {@inheritDoc} */
	@Override
	public void refresh() {
		if(navigatorViewPart != null) {
			navigatorViewPart.refresh();
		}
	}

	/** Refreshes the viewer and reveals the given element. */
	public void refreshAndReveal(Object element) {
		if(navigatorViewPart != null) {
			navigatorViewPart.refreshAndReveal(element);
		}
	}

	/** {@inheritDoc} */
	@Override
	public Saveable[] getSaveables() {
		return saveables.values().toArray(new Saveable[saveables.size()]);
	}

	/** {@inheritDoc} */
	@Override
	public Saveable[] getActiveSaveables() {
		if(navigatorViewPart == null) {
			return new Saveable[0];
		}

		EObject selection = navigatorViewPart.getSelectedModelElement();
		if(selection != null) {
			ITopLevelElement context =
					IPersistencyService.getInstance().getTopLevelElementFor(selection);
			if(saveables.get(context) != null) {
				return new Saveable[] {saveables.get(context)};
			}
		}
		return new Saveable[0];
	}

	/**
	 * Implementation of {@link Saveable}s for {@link ITopLevelElement}s.
	 */
	private static class TopLevelElementSaveable extends Saveable {

		/** Stores the {@link ITopLevelElement}. */
		public final ITopLevelElement context;

		/** Constructor. */
		public TopLevelElementSaveable(ITopLevelElement context) {
			this.context = context;
		}

		/** {@inheritDoc} */
		@Override
		public String getName() {
			return context.getSaveableName();
		}

		/** {@inheritDoc} */
		@Override
		public String getToolTipText() {
			return "";
		}

		/** {@inheritDoc} */
		@Override
		public ImageDescriptor getImageDescriptor() {
			return null;
		}

		/** {@inheritDoc} */
		@Override
		public void doSave(IProgressMonitor monitor) throws CoreException {
			try {
				context.doSave(monitor);
			} catch(IOException e) {
				error(ToolingKernelActivator.getDefault(), "Error during save of " + getName(), e);
			}
		}

		/** {@inheritDoc} */
		@Override
		public boolean isDirty() {
			return context.isDirty();
		}

		/** {@inheritDoc} */
		@Override
		public boolean equals(Object object) {
			return object instanceof TopLevelElementSaveable &&
					((TopLevelElementSaveable)object).context == context;
		}

		/** {@inheritDoc} */
		@Override
		public int hashCode() {
			return context.hashCode();
		}
	}

	/** {@inheritDoc} */
	@Override
	public void revealModelElement(EObject modelElement) {
		navigatorViewPart.revealModelElement(modelElement);
	}

	/**
	 * Notify {@link ISaveablesLifecycleListener} about saveable change of given
	 * top-level element.
	 */
	private void saveablesChanged(int event, ITopLevelElement element, boolean force) {
		Saveable saveable = saveables.get(element);
		if(saveable != null && navigatorViewPart != null) {
			ISaveablesLifecycleListener listener =
					PlatformUI.getWorkbench().getService(ISaveablesLifecycleListener.class);
			SaveablesLifecycleEvent eventObj = new SaveablesLifecycleEvent(navigatorViewPart, event,
					new Saveable[] {saveable}, force);
			listener.handleLifecycleEvent(eventObj);
		}
	}

	/** Post {@link SaveablesLifecycleEvent#POST_OPEN}. */
	private void notifySaveablesPostOpen(ITopLevelElement element) {
		saveablesChanged(SaveablesLifecycleEvent.POST_OPEN, element, false);
	}

	/** Post {@link SaveablesLifecycleEvent#POST_CLOSE}. */
	private void notifySaveablesPostClose(ITopLevelElement element) {
		saveablesChanged(SaveablesLifecycleEvent.POST_CLOSE, element, false);
	}

	/** Post {@link SaveablesLifecycleEvent#PRE_CLOSE}. */
	private void notifySaveablesPreClose(ITopLevelElement element) {
		// force the close since the saveable is already gone
		// => do not ask to save a file that has been deleted.
		saveablesChanged(SaveablesLifecycleEvent.PRE_CLOSE, element, true);
	}

	/** Post {@link SaveablesLifecycleEvent#DIRTY_CHANGED}. */
	private void notifySaveablesDirtyChanged(ITopLevelElement element) {
		saveablesChanged(SaveablesLifecycleEvent.DIRTY_CHANGED, element, false);
	}

	/** {@inheritDoc} */
	@Override
	public void tutorialStarted(ITutorialProvider provider) {
		refreshAndReveal(ITutorialService.getInstance().getActiveTutorial().getDefinition());
	}

	/** {@inheritDoc} */
	@Override
	public void tutorialStepChanged(TutorialAtomicStep step) {
		refresh();
	}

	/** {@inheritDoc} */
	@Override
	public void tutorialStopped(ITutorialProvider provider) {
		refresh();
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public Collection<IIntrospectionItem> getIntrospectionItems() {
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return null;
	}
}
