/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal;

import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.progress.UIJob;
import org.fortiss.tooling.kernel.extension.ITutorialProvider;
import org.fortiss.tooling.kernel.extension.data.TutorialStepBase;
import org.fortiss.tooling.kernel.introspection.IIntrospectionDetailsItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.IIntrospectiveKernelService;
import org.fortiss.tooling.kernel.service.IKernelIntrospectionSystemService;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.ITutorialStepUI;
import org.fortiss.tooling.kernel.ui.extension.ITutorialUIProvider;
import org.fortiss.tooling.kernel.ui.extension.ITutorialUIWhitelistProvider;
import org.fortiss.tooling.kernel.ui.internal.editor.TutorialStepUIEditor;
import org.fortiss.tooling.kernel.ui.internal.editor.TutorialStepUIEditorInput;
import org.fortiss.tooling.kernel.ui.service.ITutorialUIService;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;

/**
 * Implementation of {@link ITutorialUIService}.
 * 
 * @author hoelzl
 */
public final class TutorialUIService
		implements ITutorialUIService, ITutorialUIWhitelistProvider, IIntrospectiveKernelService {

	/** The singleton service instance. */
	private static TutorialUIService INSTANCE = new TutorialUIService();

	/** Returns the service instance. */
	public static TutorialUIService getInstance() {
		return INSTANCE;
	}

	/** The map from tutorial providers to their UI counter-part. */
	private final Map<Class<? extends ITutorialProvider>, Class<? extends ITutorialUIProvider>> providerMap =
			new HashMap<Class<? extends ITutorialProvider>, Class<? extends ITutorialUIProvider>>();

	/** The map of tutorial UI provider instances. */
	private final Map<Class<? extends ITutorialUIProvider>, ITutorialUIProvider> instanceMap =
			new HashMap<Class<? extends ITutorialUIProvider>, ITutorialUIProvider>();

	/** The list of tutorials picked up from the Eclipse install location. */
	private final Collection<Bundle> locallyInstalledTutorials = new ArrayList<Bundle>();

	/** Initializes the service. */
	public void initializeService() {
		// nothing to do here
	}

	/** Starts the service. */
	public void startService() {
		IKernelIntrospectionSystemService.getInstance().registerService(this);
		findAndStartTutorialBundles();
	}

	/**
	 * Searches the installation folder for tutorial bundles and installs and starts them using the
	 * OSGi API.
	 */
	private void findAndStartTutorialBundles() {
		try {
			// construct installation folder location
			URL loc = Platform.getInstallLocation().getURL();
			URL tutorialFolder = new URL(loc, "tutorials");
			File dir = new File(tutorialFolder.getFile());
			System.out.println("[TUIS] Searching tutorials in " + tutorialFolder.toString());
			if(!dir.isDirectory()) {
				System.out.println("[TUIS] Tutorials folder not existent or not a directory." +
						" Aborting search. No tutorials loaded.");
				return;
			}
			// search location for .jar files
			File[] tutoDirs = dir.listFiles(f -> f.isDirectory() && containsMetaInfAndManifest(f));
			if(tutoDirs.length == 0) {
				System.out.println("[TUIS] No tutorial directories found. Aborting.");
				return;
			}
			// load jar files
			BundleContext bundleContext =
					ToolingKernelUIActivator.getDefault().getBundle().getBundleContext();
			for(File tutorialDir : tutoDirs) {
				URI uri = tutorialDir.toURI();
				System.out.println("[TUIS] Loading tutorial plugin: " + uri.toString());
				try {
					Bundle tutorialBundle = bundleContext.installBundle(uri.toString());
					tutorialBundle.start();
					locallyInstalledTutorials.add(tutorialBundle);
					System.out.println("[TUIS] Successfully started: " + uri.toString());
				} catch(BundleException bundleX) {
					System.err.println("[TUIS] " + bundleX.getMessage());
				}
			}
		} catch(MalformedURLException urlEx) {
			System.err.println("[TUIS] " + urlEx.getMessage());
		}

		// start tutorial immediately if command line arguments have been given
		Display display = Display.getCurrent();
		if(display == null) {
			// Stop here if display is still null
			// Then no tutorial is started on startup of AF3
			return;
		}
		UIJob uiJob = new UIJob(display, "TutorialJob") {

			@Override
			public IStatus runInUIThread(IProgressMonitor monitor) {

				final String getenv = System.getProperty("sun.java.command");
				List<String> split = Arrays.asList(getenv.split(" "));
				// search for Tutorial
				Optional<String> cmdLineArgument =
						split.stream().filter(s -> s.contains("tutorial")).findFirst();

				if(!cmdLineArgument.isPresent()) {
					return Status.CANCEL_STATUS;
				}

				// 'tutorial = <Class name of tutorial root provider>'
				String cmdLineString = cmdLineArgument.get();
				String[] split2 = cmdLineString.split("=");
				String className = split2[split2.length - 1];

				Optional<Entry<Class<? extends ITutorialProvider>, Class<? extends ITutorialUIProvider>>> tutorialProvider =
						providerMap.entrySet().stream()
								.filter(c -> c.getKey().toString().contains(className)).findFirst();

				if(!tutorialProvider.isPresent()) {
					return Status.CANCEL_STATUS;
				}

				Class<? extends ITutorialProvider> clazz = tutorialProvider.get().getKey();
				display.asyncExec(() -> ITutorialService.getInstance().startTutorial(clazz));

				return Status.OK_STATUS;
			}
		};

		uiJob.schedule();
	}

	/** Tests for existence of META-INF sub-directory. */
	private static final boolean containsMetaInfAndManifest(File dir) {
		File meta = new File(dir, "META-INF");
		File manifest = new File(meta, "MANIFEST.MF");
		return meta.isDirectory() && manifest.isFile();
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionLabel() {
		return "Tutorial UI Service";
	}

	/** {@inheritDoc} */
	@Override
	public String getIntrospectionDescription() {
		return getIntrospectionLabel() +
				"\n\nThis service manages the user interface part of the tutorial service." +
				"\nIt provides information about elements visible in the navigator, whether the default" +
				"\nactions are enabled, and which context menu entries are visible.";
	}

	/** {@inheritDoc} */
	@Override
	public void registerTutorialUIProvider(Class<? extends ITutorialProvider> nonUIprovider,
			Class<? extends ITutorialUIProvider> uiProvider) {
		if(!providerMap.containsKey(nonUIprovider)) {
			try {
				providerMap.put(nonUIprovider, uiProvider);
				instanceMap.put(uiProvider, uiProvider.newInstance());
			} catch(Exception e) {
				error(ToolingKernelUIActivator.getDefault(),
						"Failed to instantiate tutorial UI provider " + uiProvider.getName() + "!",
						e);
			}
		} else {
			StringBuilder sb = new StringBuilder();
			sb.append("TutorialProvider ").append(nonUIprovider.getName());
			sb.append(" already mapped to ").append(uiProvider.getName());
			sb.append("!");
			error(ToolingKernelUIActivator.getDefault(), sb.toString());
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean elementVisibleInNavigator(EObject element) {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.elementVisibleInNavigator(element)) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean globalCopyPasteActionsVisible() {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.globalCopyPasteActionsVisible()) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean globalUndoRedoActionsVisible() {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.globalUndoRedoActionsVisible()) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean globalDeleteActionVisible() {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.globalDeleteActionVisible()) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean globalRenameActionVisible() {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.globalRenameActionVisible()) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean globalSelectAllActionVisible() {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.globalSelectAllActionVisible()) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean contextMenuContributionVisible(EObject element, IContributionItem contribution) {
		TutorialStepBase step = ITutorialService.getInstance().getActiveTutorialStep();
		while(step != null) {
			if(step instanceof ITutorialUIWhitelistProvider) {
				ITutorialUIWhitelistProvider p = (ITutorialUIWhitelistProvider)step;
				if(p.contextMenuContributionVisible(element, contribution)) {
					return true;
				}
			}
			step = step.getParent();
		}
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public void openInEditor(ITutorialStepUI selected) {
		try {
			IEditorInput input = new TutorialStepUIEditorInput(selected);
			IWorkbenchWindow activeWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
			String name = TutorialStepUIEditor.class.getName();
			activeWindow.getActivePage().openEditor(input, name, true);
		} catch(PartInitException pex) {
			error(ToolingKernelUIActivator.getDefault(), pex.getMessage(), pex);
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean showInIntrospectionNavigation() {
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public Collection<IIntrospectionItem> getIntrospectionItems() {
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public IIntrospectionDetailsItem getDetailsItem() {
		return null;
	}

	/** Stops the bundles installed and started from the local installation directory. */
	public void stopLocallyStartedTutorialBundles() {
		for(Bundle b : locallyInstalledTutorials) {
			try {
				b.stop();
				b.uninstall();
			} catch(BundleException e) {
				e.printStackTrace();
			}
		}
	}
}
