/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.actions;

import static org.eclipse.ui.PlatformUI.getWorkbench;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.error;
import static org.fortiss.tooling.kernel.utils.ResourceUtils.getIFile;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IConnectionCompositorService;
import org.fortiss.tooling.kernel.service.IElementCompositorService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.base.EObjectActionBase;
import org.fortiss.tooling.kernel.ui.internal.editor.ExtendableMultiPageEditor;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;

/**
 * The {@link DeleteAction} uses the {@link IElementCompositorService} and the
 * {@link IConnectionCompositorService} to determine the enabled state of this
 * action.
 * 
 * @author hoelzl
 */
public class DeleteAction extends EObjectActionBase<EObject> {

	/**
	 * {@link Job} specialization used to perform the potentially long-running deletion of an
	 * {@link ITopLevelElement}.
	 */
	private static final class DeleteTopLevelElementJob extends Job {

		/**
		 * {@link ITopLevelElement} which (together with its directly or indirectly referencing
		 * {@link Resource}s) should be deleted.
		 */
		private final ITopLevelElement topElement;

		/** {@link Shell} used to display dialog. */
		private Shell sh;

		/**
		 * Creates a {@link DeleteTopLevelElementJob} that deletes the given
		 * {@link ITopLevelElement} (together with its directly or indirectly referencing
		 * {@link Resource}s).
		 */
		public DeleteTopLevelElementJob(ITopLevelElement topElement) {
			super("Deleting \'" + topElement.getSaveableName() + "\'");
			this.topElement = topElement;
			sh = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
		}

		/** {@inheritDoc} */
		@Override
		protected IStatus run(IProgressMonitor progressMonitor) {
			final MessageDialog dialog = new MessageDialog(sh, "Confirm Delete", null,
					"Are you sure you want to delete '" + topElement.getSaveableName() + "'?",
					MessageDialog.QUESTION,
					new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 0) {
				@Override
				protected int getShellStyle() {
					return super.getShellStyle() | SWT.SHEET;
				}
			};

			sh.getDisplay().syncExec(new Runnable() {
				@Override
				public void run() {
					dialog.open();
				}
			});

			if(dialog.getReturnCode() == 0) {
				IContainer parent =
						getIFile(topElement.getRootModelElement().eResource()).getParent();

				// Close all opened views belonging to the deleted project
				IModelEditorBindingService.getInstance()
						.closeEditors(topElement.getRootModelElement());

				// Deletes the entire resource set, including its external models
				topElement.delete();

				try {
					parent.refreshLocal(IResource.DEPTH_INFINITE, progressMonitor);
				} catch(CoreException e) {
					error(ToolingKernelUIActivator.getDefault(), e.getMessage(), e);
				}
			}
			return Status.OK_STATUS;
		}
	}

	/** Flag if currently a delete operation is performed. */
	private static boolean deleteOperationInProgress = false;

	/** Constructor. */
	public DeleteAction() {
		super("Delete", PlatformUI.getWorkbench().getSharedImages()
				.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));
		setId(ActionFactory.DELETE.getId());
	}

	/** {@inheritDoc} */
	@Override
	public void run() {
		doDelete(getTargets());
	}

	/** Deletes the given targets (organized in a map by topLevel element). */
	public static void doDelete(List<EObject> unorganizedTargets) {
		Map<ITopLevelElement, List<EObject>> topLevelTargetsMap =
				topLevelMap(filterTargets(unorganizedTargets));
		for(ITopLevelElement topLevel : topLevelTargetsMap.keySet()) {
			final List<EObject> targets = topLevelTargetsMap.get(topLevel);
			boolean topLevelDeleted = false;
			for(EObject target : targets) {
				if(IPersistencyService.getInstance().isTopLevelElement(target)) {
					doTopLevelElementDelete(target);
					topLevelDeleted = true;
				}
			}
			if(!topLevelDeleted) {
				ICommandStackService.getInstance().runAsCommand(topLevel.getRootModelElement(),
						new Runnable() {
							@Override
							public void run() {
								doCompositorBasedDelete(targets);
							}
						});
			}
		}
	}

	/** Computes enabled state of the delete action. */
	@Override
	protected boolean computeEnabled() {
		if(!isGlobalDeleteActionVisible()) {
			return false;
		}
		return !deleteOperationInProgress && getTargets().stream().anyMatch(t -> canDelete(t));
	}

	/** Compute if one given target can be deleted. */
	private boolean canDelete(EObject target) {

		IWorkbenchPart activePart =
				getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart();
		if(activePart instanceof ExtendableMultiPageEditor) {
			EObject editedObject = ((ExtendableMultiPageEditor)activePart).getEditedObject();
			if(editedObject.equals(target))
				return false;
		}

		if(!IPersistencyService.getInstance().isTopLevelElement(target)) {
			// use composition services for deletion
			boolean canDecompose = IElementCompositorService.getInstance().canDecompose(target);
			return(canDecompose ||
					IConnectionCompositorService.getInstance().canDisconnect(target));
		}
		// use top-level element deletion mechanism
		ITopLevelElement topElement =
				IPersistencyService.getInstance().getTopLevelElementFor(target);
		return topElement.canDelete();
	}

	/** Executes the delete using the correct service. */
	private static void doCompositorBasedDelete(List<EObject> targets) {
		// Sort list of deletion targets such that object whose eContainer() is not contained in
		// 'targets' are deleted first (e.g., free connectors in hierarchical element models).
		Set<EObject> targetSet = new HashSet<EObject>();
		targetSet.addAll(targets);

		// First, put all objects whose container is contained in 'targets' as well as the container
		// itself to the sorted list.
		LinkedList<EObject> sortedTargets = new LinkedList<EObject>();
		for(EObject obj : targets) {
			EObject container = obj.eContainer();
			if(targetSet.contains(container)) {
				sortedTargets.addLast(obj);
				targetSet.remove(obj);
				sortedTargets.addLast(container);
				targetSet.remove(container);
			}
		}

		// Then, put all remaining objects (e.g., whose container is not contained in 'targets'), to
		// the beginning of the sorted list
		for(EObject obj : targetSet) {
			sortedTargets.addFirst(obj);
		}

		for(EObject target : sortedTargets) {
			if(IElementCompositorService.getInstance().canDecompose(target)) {
				IElementCompositorService.getInstance().decompose(target);
			} else if(IConnectionCompositorService.getInstance().canDisconnect(target)) {
				IConnectionCompositorService.getInstance().disconnect(target);
			}
		}
	}

	/** User interaction before deleting a top-level element. */
	private static void doTopLevelElementDelete(EObject target) {
		ITopLevelElement topElement =
				IPersistencyService.getInstance().getTopLevelElementFor(target);

		deleteOperationInProgress = true;
		DeleteTopLevelElementJob job = new DeleteTopLevelElementJob(topElement);
		job.schedule();

		job.addJobChangeListener(new JobChangeAdapter() {
			/** {@inheritDoc} */
			@Override
			public void done(IJobChangeEvent event) {
				deleteOperationInProgress = false;
			}
		});
	}

	/**
	 * Returns a map grouping the given model elements by top-level element,
	 * i.e., given a set of model elements M, returns a map f s.t., for every top level element top,
	 * f(top) = { m in M | top is the top level element of m }
	 */
	public static Map<ITopLevelElement, List<EObject>> topLevelMap(List<EObject> modelElements) {
		HashMap<ITopLevelElement, List<EObject>> res =
				new HashMap<ITopLevelElement, List<EObject>>();
		for(EObject eo : modelElements) {
			ITopLevelElement top = IPersistencyService.getInstance().getTopLevelElementFor(eo);
			List<EObject> elts = res.get(top);
			List<EObject> elts2;
			if(elts == null) {
				elts2 = Arrays.asList(eo);
			} else {
				elts2 = new ArrayList<EObject>(elts);
				elts2.add(eo);
			}
			res.put(top, elts2);
		}
		return res;
	}

	/**
	 * Filter the targets in order to avoid deleting not on purpose the object currently edited.
	 * See issue 1493 for more details.
	 */
	private static List<EObject> filterTargets(List<EObject> targets) {
		IWorkbenchPart activeEditor =
				getWorkbench().getActiveWorkbenchWindow().getActivePage().getActivePart();
		if(!(activeEditor instanceof ExtendableMultiPageEditor))
			return targets;
		List<EObject> res = new ArrayList<EObject>();
		EObject editedObject = ((ExtendableMultiPageEditor)activeEditor).getEditedObject();
		for(EObject target : targets) {
			if(target != editedObject) {
				res.add(target);
			}
		}
		return res;
	}
}
