/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.actions;

import static org.eclipse.ui.IPageLayout.ID_PROP_SHEET;
import static org.eclipse.ui.PlatformUI.getWorkbench;
import static org.fortiss.tooling.kernel.utils.LoggingUtils.info;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.views.properties.PropertySheet;
import org.fortiss.tooling.kernel.model.INamedElement;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.base.EObjectActionBase;

/**
 * Action for renaming the currently selected model element.
 * 
 * @author barner
 */
public class RenameAction extends EObjectActionBase<EObject> {

	/** Constructor. */
	public RenameAction() {
		super("Rename", ToolingKernelUIActivator.getImageDescriptor("icons/pencil.png"));
		setId(ActionFactory.RENAME.getId());
	}

	/** Flag if currently a {@link RenameAction} is in progress. */
	private boolean isRenameInProgress = false;

	/** {@inheritDoc} */
	@Override
	public void run() {
		try {
			setRenameIsInProgress(true);

			// Raise properties view
			IWorkbenchWindow activeWorkbenchWindow = getWorkbench().getActiveWorkbenchWindow();
			IWorkbenchPage activePage = activeWorkbenchWindow.getActivePage();
			activePage.showView(ID_PROP_SHEET, null, IWorkbenchPage.VIEW_VISIBLE);

			// Reset current selection if it is identically to the target object of this action
			// (otherwise, the next statement would be a NOP).
			ISelection selection = activeWorkbenchWindow.getSelectionService().getSelection();
			Object currentlySelectedElement = null;
			if(selection instanceof StructuredSelection) {
				currentlySelectedElement = ((StructuredSelection)selection).getFirstElement();
			}
			PropertySheet propertySheet = (PropertySheet)activePage.findView(ID_PROP_SHEET);
			if(currentlySelectedElement == getTarget()) {
				propertySheet.selectionChanged(activePage.getActivePart(),
						new StructuredSelection());
			}

			// Ensure that the target object of this rename action gets the selection (in order to
			// enable the user to edit it right away).
			Object[] targetElements = {getTarget()};
			propertySheet.selectionChanged(activePage.getActivePart(),
					new StructuredSelection(targetElements));
		} catch(PartInitException e) {
			info(ToolingKernelUIActivator.getDefault(), "No property section to display.");
		} finally {
			setRenameIsInProgress(false);
		}
	}

	/** {@inheritDoc} */
	@Override
	protected boolean computeEnabled() {
		return getTarget() instanceof INamedElement;
	}

	/** Sets {@link #isRenameInProgress}. */
	private synchronized void setRenameIsInProgress(boolean isRenameInProgress) {
		this.isRenameInProgress = isRenameInProgress;
	}

	/** Returns {@link #isRenameInProgress}. */
	public synchronized boolean isRenameInProgress() {
		return isRenameInProgress;
	}
}
