/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.actions;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.fortiss.tooling.kernel.ui.util.SelectionUtils.setSelection;

import java.util.List;

import org.conqat.ide.commons.ui.ui.WorkbenchUtils;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.ActionFactory;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.base.EObjectActionBase;
import org.fortiss.tooling.kernel.ui.internal.editor.ExtendableMultiPageEditor;
import org.fortiss.tooling.kernel.ui.internal.views.NavigatorViewPart;

/**
 * Action for selecting all elements in the currently active editor. the currently selected model
 * element.
 * 
 * @author barner
 */
public class SelectAllAction extends EObjectActionBase<EObject> {

	/** Constructor. */
	public SelectAllAction() {
		super("Select all", ToolingKernelUIActivator.getImageDescriptor("icons/select_all.png"));
		setId(ActionFactory.SELECT_ALL.getId());
	}

	/** {@inheritDoc} */
	@Override
	public void run() {

		EObject target = getTarget();
		if(target == null) {
			return;
		}
		IWorkbenchPart activePart = WorkbenchUtils.getActiveWorkbenchPage().getActivePart();
		List<EObject> objs = emptyList();
		if(activePart instanceof ExtendableMultiPageEditor) {
			// If an editor is active editor, select all children of the currently edited object
			objs = target.eContents();
		} else if(activePart instanceof NavigatorViewPart) {
			// If the model navigator is active ...
			if(target.eContainer() != null) {
				// ... select all siblings of the currently selected object,
				objs = target.eContainer().eContents();
			} else {
				// ... or the root elements of all projects in case the root element of a
				// project is selected
				List<ITopLevelElement> topLevelElems =
						IPersistencyService.getInstance().getTopLevelElements();
				objs = topLevelElems.stream().map(t -> t.getRootModelElement()).collect(toList());
			}
		}

		setSelection(objs);
	}
}
