/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.databinding;

import org.eclipse.core.databinding.observable.Diffs;
import org.eclipse.core.databinding.observable.Realm;
import org.eclipse.core.databinding.observable.value.AbstractObservableValue;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.swt.widgets.Display;
import org.fortiss.tooling.kernel.service.ICommandStackService;

/**
 * Implementation of {@link IObservableValue} for {@link EObject}s.
 * 
 * @author hummel
 */
public class EObjectObservableValue extends AbstractObservableValue<Object> {

	/** The wrapped object. */
	private final EObject eObject;

	/** The wrapped structural feature. */
	private final EStructuralFeature structuralFeature;

	/** Flag used for detecting our own updates. */
	private boolean updating = false;

	/** The adapter for sending change events. */
	private final Adapter modelChangeAdapter = new AdapterImpl() {
		/** {@inheritDoc} */
		@Override
		public void notifyChanged(final Notification msg) {
			if(!updating && msg.getFeature() == structuralFeature) {
				Display.getDefault().asyncExec(new Runnable() {
					@Override
					public void run() {
						fireValueChange(
								Diffs.createValueDiff(msg.getOldValue(), msg.getNewValue()));
					}
				});
			}
		}
	};

	/** Constructor. */
	public EObjectObservableValue(Realm realm, EObject eObject,
			EStructuralFeature structuralFeature) {
		super(realm);

		if(eObject == null)
			throw new NullPointerException();
		if(structuralFeature == null)
			throw new NullPointerException();

		this.eObject = eObject;
		this.structuralFeature = structuralFeature;

		eObject.eAdapters().add(modelChangeAdapter);
	}

	/** {@inheritDoc} */
	@Override
	public synchronized void dispose() {
		eObject.eAdapters().remove(modelChangeAdapter);
		super.dispose();
	}

	/** {@inheritDoc} */
	@Override
	protected Object doGetValue() {
		return eObject.eGet(structuralFeature);
	}

	/** {@inheritDoc} */
	@Override
	protected void doSetValue(final Object value) {
		Object oldValue = doGetValue();
		if(oldValue == null && value == null || oldValue != null && oldValue.equals(value)) {
			return;
		}

		updating = true;
		ICommandStackService.getInstance().runAsCommand(eObject, new Runnable() {
			@Override
			public void run() {
				eObject.eSet(structuralFeature, value);
			}
		});
		updating = false;
	}

	/** {@inheritDoc} */
	@Override
	public Object getValueType() {
		if(structuralFeature.isMany()) {
			return EList.class;
		}

		if(structuralFeature instanceof EAttribute) {
			return ((EAttribute)structuralFeature).getEAttributeType().getInstanceClass();
		}
		return ((EReference)structuralFeature).getEReferenceType().getInstanceClass();
	}
}
