/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.editor;

import static org.conqat.ide.commons.ui.logging.LoggingUtils.error;

import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.EventObject;
import java.util.List;

import org.conqat.lib.commons.collections.IdentityHashSet;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.ISaveablePart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.tabbed.ITabbedPropertySheetPageContributor;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.fortiss.tooling.kernel.ToolingKernelActivator;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.ui.extension.IModelEditor;
import org.fortiss.tooling.kernel.ui.extension.IModelEditorBinding;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.extension.base.LWFXEFEditorBase;
import org.fortiss.tooling.kernel.ui.listener.ExtendableMultiPageEditorPageChangeListener;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;
import org.fortiss.tooling.kernel.ui.util.PropertiesConstantUtils;

/**
 * This editor is used for displaying multiple editors provided by editor
 * bindings for a given model element.
 * 
 * @author hoelzl
 */
@SuppressWarnings("rawtypes")
public class ExtendableMultiPageEditor extends MultiPageEditorPart
		implements ITabbedPropertySheetPageContributor, CommandStackListener, ISaveablePart,
		IModelEditor<EObject> {
	/** The editor's ID. */
	public static String ID = ExtendableMultiPageEditor.class.getName();

	/**
	 * The object shown in this editor. This is valid as soon as
	 * {@link #init(IEditorSite, IEditorInput)} has been called.
	 */
	protected EObject editedObject;

	/** The model element handler to be used with the {@link #editedObject}. */
	protected IModelElementHandler<EObject> handler;

	/**
	 * The adapter used for learning about changes in the {@link #editedObject}.
	 */
	private final Adapter editedObjectChangeAdapter = new AdapterImpl() {

		/** {@inheritDoc} */
		@Override
		public void notifyChanged(Notification msg) {
			editedObjectChanged();
		}
	};

	/** Stores the binding editor listeners. */
	private final Collection<ExtendableMultiPageEditorPageChangeListener> bindingEditorListeners =
			new IdentityHashSet<ExtendableMultiPageEditorPageChangeListener>();

	/** Adds a page change listener. */
	public void addBindingEditorListener(ExtendableMultiPageEditorPageChangeListener listener) {
		bindingEditorListeners.add(listener);
	}

	/** Removes a page change listener. */
	public void removeBindingEditorListener(ExtendableMultiPageEditorPageChangeListener listener) {
		bindingEditorListeners.remove(listener);
	}

	/** {@inheritDoc} */
	@Override
	protected void pageChange(int newPageIndex) {
		super.pageChange(newPageIndex);

		for(ExtendableMultiPageEditorPageChangeListener listener : bindingEditorListeners) {
			listener.pageChanged();
		}
	}

	/** {@inheritDoc} */
	@Override
	public void setFocus() {
		super.setFocus();
		// Workaround to prevent tab icons from disappearing when tabs are reordered
		firePropertyChange(IWorkbenchPart.PROP_TITLE);
	}

	/**
	 * This is called whenever something about the currently edited object
	 * changes. This is used to update the part name and to close the editor if
	 * the object is removed.
	 */
	protected void editedObjectChanged() {
		setPartName(handler.getName(editedObject));
		setContentDescription(handler.getDescription(editedObject));

		if(editedObject.eContainer() == null) {
			getSite().getPage().closeEditor(this, false);
		}
	}

	/** Returns the active editor. */
	@SuppressWarnings("unchecked")
	public IModelEditor<EObject> getActiveModelEditor() {
		return (IModelEditor<EObject>)getActiveEditor();
	}

	/** {@inheritDoc} */
	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		if(!(input instanceof ModelElementEditorInput)) {
			throw new PartInitException(
					"BindingEditor initialized with wrong editor input object.");
		}
		super.init(site, input);

		ModelElementEditorInput meInput = ((ModelElementEditorInput)input);
		editedObject = meInput.getModelElement();
		if(editedObject == null) {
			throw new PartInitException("Missing model element!");
		}

		handler = meInput.getModelElementHandler();
		if(handler == null) {
			throw new PartInitException("Missing model element handler!");
		}

		String name = handler.getName(editedObject);
		if(name == null) {
			name = "";
		}
		setPartName(name);

		String desc = handler.getDescription(editedObject);
		if(desc == null) {
			desc = "";
		}
		setContentDescription(desc);

		setTitleImage(handler.getIcon(editedObject));

		editedObject.eAdapters().add(editedObjectChangeAdapter);

		ITopLevelElement target =
				IPersistencyService.getInstance().getTopLevelElementFor(editedObject);
		ICommandStackService.getInstance().addCommandStackListener(target, this);
	}

	/** {@inheritDoc} */
	@Override
	protected void createPages() {
		int pageIndex = 0;
		List<IModelEditorBinding<EObject>> bindings =
				IModelEditorBindingService.getInstance().getBindings(editedObject);
		for(IModelEditorBinding<EObject> editorBinding : bindings) {
			try {
				Class<? extends IEditorPart> editorClass =
						editorBinding.getEditorClass(editedObject);
				boolean expertViewIsActive = INavigatorService.getInstance().isExpertViewActive();
				if(editorClass != null &&
						(expertViewIsActive || !editorBinding.hiddenInNonExpertView())) {
					Class<? extends EObject> inputType = editedObject.getClass();
					IEditorPart editorPart =
							constructEditorPart(editorClass, inputType, editorBinding);
					addPage(editorPart, getEditorInput());
					setPageText(pageIndex++, editorBinding.getLabel(editedObject));
				}
			} catch(Exception ex) {
				error(ToolingKernelActivator.getDefault(), "Editor instantiation failed.", ex);
			}
		}
	}

	/**
	 * Constructs an {@link IEditorPart} instance of the given {@code editorClass}.
	 * {@link IEditorPart}s that take the input model element type as a parameter are preferred over
	 * no-arg constructors. A preferred {@link IModelEditorBinding} can be provided that is
	 * currently used by {@link LWFXEFEditorBase}-based editors to lookup JavaFX factories.
	 */
	protected IEditorPart constructEditorPart(Class<? extends IEditorPart> editorClass,
			Class<? extends EObject> inputType, IModelEditorBinding<EObject> editorBinding)
			throws Exception {
		IEditorPart newEditorPart = null;
		try {
			Constructor<? extends IEditorPart> ctor = editorClass.getConstructor(Class.class);
			newEditorPart = ctor.newInstance(inputType);
		} catch(NoSuchMethodException | SecurityException e) {
			// Fallback for no-arg constructors.
			Constructor<? extends IEditorPart> ctor = editorClass.getConstructor();
			newEditorPart = ctor.newInstance();
		}

		if(newEditorPart instanceof LWFXEFEditorBase) {
			((LWFXEFEditorBase<?>)newEditorPart).setEditorBinding(editorBinding);
		}
		return newEditorPart;
	}

	/** {@inheritDoc} */
	@Override
	public void doSave(IProgressMonitor monitor) {
		ITopLevelElement topElement =
				IPersistencyService.getInstance().getTopLevelElementFor(editedObject);
		try {
			topElement.doSave(monitor);
		} catch(Exception e) {
			error(ToolingKernelActivator.getDefault(),
					"Error during save of " + topElement.getSaveableName(), e);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void doSaveAs() {
		// not needed
	}

	/** {@inheritDoc} */
	@Override
	public void dispose() {
		ITopLevelElement target =
				IPersistencyService.getInstance().getTopLevelElementFor(editedObject);
		ICommandStackService.getInstance().removeCommandStackListener(target, this);
		editedObject.eAdapters().remove(editedObjectChangeAdapter);
		super.dispose();
	}

	/** {@inheritDoc} */
	@Override
	public boolean isDirty() {
		return ICommandStackService.getInstance().isDirty(editedObject);
	}

	/** {@inheritDoc} */
	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isSaveOnCloseNeeded() {
		return false;
	}

	/** {@inheritDoc} */
	@Override
	public String getContributorId() {
		return PropertiesConstantUtils.TABBED_PROPERTY_CONTRIBUTOR_ID;
	}

	/** {@inheritDoc} */
	@SuppressWarnings("unchecked")
	@Override
	public <S> S getAdapter(Class<S> adapter) {
		if(adapter == IPropertySheetPage.class) {
			return (S)new TabbedPropertySheetPage(this);
		}
		return super.getAdapter(adapter);
	}

	/** {@inheritDoc} */
	@Override
	public void commandStackChanged(EventObject event) {
		getEditorSite().getWorkbenchWindow().getShell().getDisplay().syncExec(new Runnable() {
			/** {@inheritDoc} */
			@Override
			public void run() {
				firePropertyChange(IWorkbenchPartConstants.PROP_DIRTY);
			}
		});
	}

	/**
	 * Finds a contributed editor of the given class in the list of editor
	 * pages.
	 */
	@SuppressWarnings("unchecked")
	public <T extends EObject, E extends IModelEditor<T>> E getPageEditor(Class<E> editorClass) {
		int num = getPageCount();
		for(int i = 0; i < num; i++) {
			IEditorPart part = getEditor(i);
			if(editorClass.isInstance(part)) {
				return (E)part;
			}
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	public EObject getEditedObject() {
		return editedObject;
	}

	/** {@inheritDoc} */
	@Override
	public void navigateTo(EObject element) {
		// delegate to active editor
		IEditorPart part = getActiveEditor();
		if(part instanceof IModelEditor) {
			((IModelEditor)part).navigateTo(element);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void setHighlight(EObject element, boolean highlighted) {
		// delegate to active editor
		IEditorPart part = getActiveEditor();
		if(part instanceof IModelEditor) {
			((IModelEditor)part).setHighlight(element, highlighted);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void clearAllHighlights() {
		// delegate to active editor
		IEditorPart part = getActiveEditor();
		if(part instanceof IModelEditor) {
			((IModelEditor)part).clearAllHighlights();
		}
	}
}
