/*-----------------------------------------------------------------------+
 | org.fortiss.tooling.kernel.ui.internal.views                          |
 |                                                                       |
 |                                                                       |
 | Copyright (c)  2004-2008 Technische Universitaet Muenchen             |
 |                                                                       |
 | Technische Universitaet Muenchen               #########  ##########  |
 | Institut fuer Informatik - Lehrstuhl IV           ##  ##  ##  ##  ##  |
 | Prof. Dr. Manfred Broy                            ##  ##  ##  ##  ##  |
 | Boltzmannstr. 3                                   ##  ##  ##  ##  ##  |
 | 85748 Garching bei Muenchen                       ##  ##  ##  ##  ##  |
 | Germany                                           ##  ######  ##  ##  |
 +-----------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.views;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.fortiss.tooling.kernel.extension.data.Prototype;
import org.fortiss.tooling.kernel.service.ICommandStackService;
import org.fortiss.tooling.kernel.service.IElementCompositorService;
import org.fortiss.tooling.kernel.service.IPrototypeService;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.extension.IContextMenuContributor;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.extension.data.ContextMenuContextProvider;
import org.fortiss.tooling.kernel.ui.service.IContextMenuService;
import org.fortiss.tooling.kernel.ui.service.IModelEditorBindingService;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;
import org.fortiss.tooling.kernel.utils.LoggingUtils;

/**
 * Generic context menu for prototype-based addition of new elements.
 * 
 * @author hoelzl
 */
public class GenericNewMenu implements IContextMenuContributor {

	/** List of available actions. */
	private final List<AddPrototypeAction> actions = new ArrayList<AddPrototypeAction>();

	/** Constructor. */
	public GenericNewMenu() {
		for(Prototype prototype : IPrototypeService.getInstance().getPrimaryPrototypes()) {
			try {
				actions.add(new AddPrototypeAction(prototype));
			} catch(Exception e) {
				LoggingUtils.error(ToolingKernelUIActivator.getDefault(),
						"Unexpected exception while adding prototypes", e);
			}
		}
	}

	/** Returns the contributed items, depending on whether the expert mode is active or not. */
	protected List<IContributionItem> getContributedItems(EObject selectedObject,
			boolean isExpertModeActive) {
		List<IContributionItem> contributionItems = new ArrayList<IContributionItem>();
		for(AddPrototypeAction apa : actions) {
			if(apa.prepare(selectedObject, isExpertModeActive)) {
				contributionItems.add(new ActionContributionItem(apa));
			}
		}
		return contributionItems;
	}

	/** {@inheritDoc} */
	@Override
	public List<IContributionItem> getContributedItems(EObject selectedObject,
			ContextMenuContextProvider contextProvider) {
		return getContributedItems(selectedObject, false);
	}

	/** {@inheritDoc} */
	@Override
	public String getMenuSectionID() {
		return IContextMenuService.TOP_MOST_MENU_SECTION_ID;
	}

	/** Action for creating a new prototype. */
	public static class AddPrototypeAction extends Action {

		/** The prototype. */
		private final Prototype prototype;

		/** The container. */
		private EObject container;

		/** Constructor. */
		public AddPrototypeAction(Prototype prototype) {
			super(prototype.getName(), getIcon(prototype));
			this.prototype = prototype;
		}

		/** Get the icon of the prototype */
		public static ImageDescriptor getIcon(Prototype prototype) {
			EObject element = prototype.getPrototype();
			IModelElementHandler<EObject> handler =
					IModelElementHandlerService.getInstance().getModelElementHandler(element);

			if(handler == null)
				return null;

			return handler.getIconImageDescriptor(element);
		}

		/** {@inheritDoc} */
		@Override
		public void run() {
			ICommandStackService.getInstance().runAsCommand(container, new Runnable() {

				@Override
				public void run() {
					EObject newObject = prototype.getPrototypeCopy();
					if(IElementCompositorService.getInstance().compose(container, newObject,
							null)) {
						IModelEditorBindingService.getInstance().openInEditor(newObject);
					}
				}
			});
		}

		/**
		 * Stores the given container and returns whether the underlying
		 * prototype can be added to it.
		 */
		public boolean prepare(EObject container, boolean expert) {
			if(ITutorialService.getInstance().isTutorialActive()) {
				if(!ITutorialService.getInstance().prototypeActive(prototype)) {
					this.container = null;
					return false;
				}
			}
			IModelElementHandler<EObject> handler = IModelElementHandlerService.getInstance()
					.getModelElementHandler(prototype.getPrototype());
			if(!expert && handler != null && handler.hiddenInNonExpertView()) {
				this.container = null;
				return false;
			}
			this.container = container;
			return IElementCompositorService.getInstance().canCompose(container,
					prototype.getPrototype(), null);
		}

		/** Returns the associated prototype. */
		public Prototype getPrototype() {
			return prototype;
		}
	}
}
