/*-------------------------------------------------------------------------+
| Copyright 2023 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.views;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.fortiss.tooling.kernel.service.IPrototypeService.PROTOTYPE_DATA_FORMAT;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.conqat.ide.commons.ui.ui.EmptyPartListener;
import org.conqat.lib.commons.collections.IdentityHashSet;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.fx.ui.workbench3.FXViewPart;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPart;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeViewer;
import org.fortiss.tooling.kernel.extension.data.Prototype;
import org.fortiss.tooling.kernel.extension.data.Prototype.PrototypeComparator;
import org.fortiss.tooling.kernel.extension.data.PrototypeCategory;
import org.fortiss.tooling.kernel.extension.data.PrototypeCategory.PrototypeCategoryComparator;
import org.fortiss.tooling.kernel.internal.PrototypeService;
import org.fortiss.tooling.kernel.service.IPrototypeService;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.extension.base.EditorBase;
import org.fortiss.tooling.kernel.ui.internal.editor.ExtendableMultiPageEditor;
import org.fortiss.tooling.kernel.ui.listener.ExtendableMultiPageEditorPageChangeListener;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;

import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.TextField;
import javafx.scene.control.TreeItem;
import javafx.scene.input.ClipboardContent;
import javafx.scene.layout.BorderPane;

/**
 * {@link FXViewPart} for showing composable model elements with DND editing support.
 * 
 * @author hoelzl
 */
public final class ModelElementsView extends FXViewPart {

	/** A dummy object serving as hidden root element of the tree view. */
	private static final Object DUMMY_ROOT = new Object();

	/** The tree-table viewer. */
	private DynamicTreeViewer<Object> treeViewer;

	/** The filter text field if filtering is enabled in content provider. */
	private TextField filterWidget;

	/** Current active {@link ExtendableMultiPageEditor}. */
	private ExtendableMultiPageEditor activeBindingEditor = null;

	/** Stores the editor activation listener. */
	private EditorActivationListener editorActivationListener = new EditorActivationListener();

	/** The container object used. */
	private EObject containerObject = null;

	/** The prototypes supported by the current editor. */
	private Set<Prototype> supportedBaseClasses = new IdentityHashSet<Prototype>();

	/** Cached expansion status of each contained prototype category. */
	private Map<PrototypeCategory, Boolean> prototypeCategoryExpansionCache = new HashMap<>();

	/** Listener for reacting to changes of the embedded editor. */
	private final ExtendableMultiPageEditorPageChangeListener bindingEditorPageChangeListener =
			new ExtendableMultiPageEditorPageChangeListener() {
				@Override
				public void pageChanged() {
					updateEditorFilters(activeBindingEditor.getActiveModelEditor());
				}
			};

	/** {@inheritDoc} */
	@Override
	protected Scene createFxScene() {
		getSite().getWorkbenchWindow().getPartService().addPartListener(editorActivationListener);

		BorderPane pane = new BorderPane();
		ContentProvider contentProvider = new ContentProvider();
		UIProvider uiProvider = new UIProvider();

		filterWidget = new TextField();
		filterWidget.textProperty().addListener((obs, oVal, nVal) -> {
			contentProvider.setFilterExpression(nVal);
			treeViewer.update();
		});

		// Do not show dummy container.
		boolean showRoot = false;
		// All categories/elements collapsed by default.
		int revealLevel = 0;
		treeViewer = new DynamicTreeViewer<>(DUMMY_ROOT, showRoot, revealLevel, contentProvider,
				uiProvider);

		pane.setTop(filterWidget);
		pane.setCenter(treeViewer.getControl());

		return new Scene(pane);
	}

	/** {@inheritDoc} */
	@Override
	protected void setFxFocus() {
		filterWidget.requestFocus();
	}

	/** Updates the filters according to the given editor. */
	@SuppressWarnings("unchecked")
	private void updateEditorFilters(IEditorPart editor) {
		supportedBaseClasses.clear();
		if(editor instanceof EditorBase) {
			EditorBase<? extends EObject> editorBase = (EditorBase<? extends EObject>)editor;
			containerObject = editorBase.getEditedObject();

			IPrototypeService ps = IPrototypeService.getInstance();
			for(Class<? extends EObject> clazz : editorBase.getVisibleEObjectTypes()) {
				supportedBaseClasses.addAll(ps.getComposablePrototypes(clazz));
			}
		} else {
			containerObject = null;
		}

		cacheCurrentItemExpansionStatus();

		treeViewer.update();

		expandCorrectItems();
	}

	/**
	 * Iterates through the current active {@link TreeItem}s and stores/caches their current
	 * expansion status, i.e., whether they are expanded as item or not.
	 */
	private void cacheCurrentItemExpansionStatus() {
		TreeItem<Object> rootItem = treeViewer.getControl().getRoot();
		for(TreeItem<Object> topLevelItem : rootItem.getChildren()) {
			cacheItemExpansionStatusOfItemsRecursively(topLevelItem);
		}
	}

	/**
	 * Stores/caches the current expansion status of the given {@link TreeItem}, i.e., whether they
	 * are expanded as item or not, and does the same recursively for all of its (possible)
	 * children.
	 */
	private void cacheItemExpansionStatusOfItemsRecursively(TreeItem<Object> item) {
		Object itemObject = item.getValue();
		if(itemObject instanceof PrototypeCategory) {
			PrototypeCategory category = (PrototypeCategory)itemObject;
			prototypeCategoryExpansionCache.put(category, item.isExpanded());
			for(TreeItem<Object> childItem : item.getChildren()) {
				cacheItemExpansionStatusOfItemsRecursively(childItem);
			}
		}
	}

	/**
	 * Expands the {@link TreeItem}s that should be expanded due to their (prototype) attribute (if
	 * they should be expanded by default) or because they were already manually expanded by the
	 * user before. This method works on the currently contained items of the current viewer.
	 * Therefore, the method should only be called after the viewer was updated (when its items are
	 * the currently wanted ones).
	 */
	private void expandCorrectItems() {
		TreeItem<Object> rootItem = treeViewer.getControl().getRoot();
		for(TreeItem<Object> topLevelItem : rootItem.getChildren()) {
			expandItemsRecursivelyFromCache(topLevelItem);
		}
	}

	/**
	 * Expands the given {@link TreeItem} if it represents a {@link PrototypeCategory} that should
	 * be expanded. In case of expansion, continues checking the same for all the children of this
	 * category (recursively).
	 */
	private void expandItemsRecursivelyFromCache(TreeItem<Object> item) {
		Object itemObject = item.getValue();
		if(itemObject instanceof PrototypeCategory) {
			PrototypeCategory category = (PrototypeCategory)itemObject;
			// Try getting previous/cached status before checking default one.
			Boolean shouldBeExpanded = prototypeCategoryExpansionCache.get(category);
			if(shouldBeExpanded == null) {
				shouldBeExpanded = category.getExpandedByDefaultStatus();
			}
			if(shouldBeExpanded) {
				// Expand this category (must also be set for all its direct children, otherwise the
				// children will not show up in the expansion).
				treeViewer.expandItem(item);
				for(TreeItem<Object> childItem : item.getChildren()) {
					treeViewer.expandItem(childItem);
					expandItemsRecursivelyFromCache(childItem);
				}
			}
		}
	}

	/** Switches to the given workbench editor part. */
	private void switchWorkbenchEditor(IEditorPart editor) {
		// Trigger an update of all prototypes, especially for the reuse library, because they can
		// change dynamically during runtime. Needed for the correct model element view.
		PrototypeService.getInstance().updatePrototypes();

		if(activeBindingEditor != null) {
			activeBindingEditor.removeBindingEditorListener(bindingEditorPageChangeListener);
			activeBindingEditor = null;
		}

		if(editor instanceof ExtendableMultiPageEditor) {
			activeBindingEditor = (ExtendableMultiPageEditor)editor;
			activeBindingEditor.addBindingEditorListener(bindingEditorPageChangeListener);
			updateEditorFilters(activeBindingEditor.getActiveModelEditor());
		} else if(editor != null) {
			updateEditorFilters(editor);
		}
	}

	/** {@link DynamicTreeContentProviderBase} for the model element library view. */
	private class ContentProvider extends DynamicTreeContentProviderBase<Object> {
		/** {@inheritDoc} */
		@Override
		protected Collection<? extends Object> getChildren(Object parent) {
			if(parent instanceof PrototypeCategory) {
				PrototypeCategory protoCat = (PrototypeCategory)parent;
				return asList(protoCat.getChildren());
			}
			if(parent == DUMMY_ROOT) {
				return IPrototypeService.getInstance().getAllTopLevelPrototypesCategories();
			}
			return emptyList();
		}

		/** {@inheritDoc} */
		@Override
		protected boolean filter(Object element, String filterValue) {
			if(containerObject == null || supportedBaseClasses.isEmpty()) {
				return false;
			}
			if(element instanceof PrototypeCategory) {
				PrototypeCategory protoCat = (PrototypeCategory)element;
				for(Object child : protoCat.getChildren()) {
					if(filter(child, filterValue)) {
						return true;
					}
				}
				return false;
			}
			if(supportedBaseClasses.contains(element)) {
				if(element instanceof Prototype) {
					boolean isExpertViewActive =
							INavigatorService.getInstance().isExpertViewActive();

					Prototype proto = (Prototype)element;
					IModelElementHandlerService mhs = IModelElementHandlerService.getInstance();
					IModelElementHandler<EObject> handler =
							mhs.getModelElementHandler(proto.getPrototype());
					if(!isExpertViewActive && handler.hiddenInNonExpertView()) {
						return false;
					}
					if(filterValue == null || "".equals(filterValue)) {
						return true;
					}
					return proto.getName().toLowerCase().contains(filterValue.toLowerCase());
				}
				throw new RuntimeException("Model Elements view: encountered unexpected type: " +
						element.getClass().getSimpleName());
			}
			return false;
		}

		/** {@inheritDoc} */
		@Override
		protected Comparator<Object> getSortingComparator() {

			/** {@link Comparator} for comparing {@link PrototypeCategory}s */
			final PrototypeCategoryComparator catCmp = new PrototypeCategoryComparator();

			/** {@link Comparator} for comparing {@link Prototype}s */
			final PrototypeComparator protoCmp = new PrototypeComparator();

			return new Comparator<Object>() {

				/** {@inheritDoc} */
				@Override
				public int compare(Object o1, Object o2) {
					if(o1 instanceof PrototypeCategory && o2 instanceof PrototypeCategory) {
						return catCmp.compare((PrototypeCategory)o1, (PrototypeCategory)o1);
					} else if(o1 instanceof Prototype && o1 instanceof Prototype) {
						return protoCmp.compare((Prototype)o1, (Prototype)o1);
					}
					// Never here
					return 0;
				}
			};
		}
	}

	/** {@link DynamicTreeUIProviderBase} for the model element library view. */
	private class UIProvider extends DynamicTreeUIProviderBase<Object> {
		/** {@inheritDoc} */
		@Override
		public String getLabel(Object element) {
			if(element instanceof Prototype) {
				Prototype proto = (Prototype)element;
				return proto.getName();
			}
			if(element instanceof PrototypeCategory) {
				PrototypeCategory protoCat = (PrototypeCategory)element;
				return protoCat.getName();
			}
			return element == null ? "" : element.toString();
		}

		/** {@inheritDoc} */
		@Override
		public Node getIconNode(Object element) {
			if(element instanceof Prototype) {
				Prototype proto = (Prototype)element;
				EObject eoPrototype = proto.getPrototype();
				return IModelElementHandlerService.getInstance().getFXIcon(eoPrototype);
			}
			if(element instanceof PrototypeCategory) {
				PrototypeCategory protoCat = (PrototypeCategory)element;
				if(protoCat.getChildren().length > 0) {
					return getIconNode(protoCat.getChildren()[0]);
				}
			}
			return null;
		}

		/** {@inheritDoc} */
		@Override
		public ClipboardContent getDragClipboardContent(Object element) {
			if(element instanceof Prototype) {
				Prototype proto = (Prototype)element;
				String uid = IPrototypeService.getInstance().getUniquePrototypeID(proto);
				if(uid != null) {
					ClipboardContent cbContent = new ClipboardContent();
					cbContent.put(PROTOTYPE_DATA_FORMAT, uid);
					return cbContent;
				}
			}
			return null;
		}
	}

	/** If an editor in a different Project is opened the Model is reinitialized */
	private class EditorActivationListener extends EmptyPartListener {
		/** Change the tree viewer content whenever workbench part changes. */
		@Override
		public void partActivated(IWorkbenchPart workbenchPart) {
			if(!(workbenchPart instanceof IEditorPart)) {
				return;
			}

			IEditorPart part = (IEditorPart)workbenchPart;
			treeViewer.update();
			switchWorkbenchEditor(part);
		}
	}
}
