/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.views;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.conqat.ide.commons.ui.jface.TreeContentProviderBase;
import org.eclipse.emf.ecore.EObject;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.extension.data.TutorialCompositeStep;
import org.fortiss.tooling.kernel.extension.data.TutorialStepBase;
import org.fortiss.tooling.kernel.model.ILibrary;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.extension.IModelElementHandler;
import org.fortiss.tooling.kernel.ui.internal.NavigatorService;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;
import org.fortiss.tooling.kernel.ui.service.ITutorialUIService;

/**
 * Content provider for the model navigator tree viewer.
 * 
 * @author hoelzlf
 */
public class NavigatorTreeContentProvider extends TreeContentProviderBase {

	/** {@inheritDoc} */
	@Override
	public Object[] getElements(Object inputElement) {
		// delegate to the top-level element contexts of the persistency service
		List<Object> result = new LinkedList<Object>();

		for(ITopLevelElement context : IPersistencyService.getInstance().getTopLevelElements()) {
			EObject root = context.getRootModelElement();
			if(!(root instanceof ILibrary)) {
				result.add(root);
			}
		}
		if(ITutorialService.getInstance().isTutorialActive()) {
			result.add(ITutorialService.getInstance().getActiveTutorial().getDefinition());
		}
		return result.toArray();
	}

	/** {@inheritDoc} */
	@Override
	public Object[] getChildren(Object parent) {
		// delegate to the model element handlers
		if(parent instanceof EObject) {
			EObject parentElement = (EObject)parent;

			IModelElementHandler<EObject> handler =
					IModelElementHandlerService.getInstance().getModelElementHandler(parentElement);
			if(handler != null) {
				List<EObject> children = new ArrayList<EObject>();
				children.addAll(filterTutorialElements(
						filterExpertElements(handler.getSubnodes(parentElement))));
				children.addAll(filterTutorialElements(
						filterExpertElements(handler.getSpecifications(parentElement))));
				return children.toArray();
			}
		}
		if(parent instanceof TutorialCompositeStep) {
			return ((TutorialCompositeStep)parent).getSteps().toArray();
		}
		return new Object[0];
	}

	/**
	 * If {@link ITutorialService#isTutorialActive()} is <code>true</code> then we filter elements,
	 * which are not whitelisted.
	 */
	private Collection<? extends EObject> filterTutorialElements(List<EObject> elements) {
		if(ITutorialService.getInstance().isTutorialActive()) {
			ITutorialUIService uiService = ITutorialUIService.getInstance();
			Predicate<? super EObject> isVisible = e -> uiService.elementVisibleInNavigator(e);
			return elements.stream().filter(isVisible).collect(Collectors.toList());
		}
		return elements;
	}

	/**
	 * If {@link INavigatorService#isExpertViewActive()} is false then we filter out elements
	 * according to {@link IModelElementHandler#hiddenInNonExpertView()}
	 */
	private List<EObject> filterExpertElements(List<EObject> elements) {
		if(!INavigatorService.getInstance().isExpertViewActive()) {
			List<EObject> filtered = new ArrayList<EObject>();
			for(EObject current : elements) {
				IModelElementHandler<EObject> handler =
						IModelElementHandlerService.getInstance().getModelElementHandler(current);
				if(handler != null && !handler.hiddenInNonExpertView()) {
					filtered.add(current);
				}
			}
			return filtered;
		}
		return elements;
	}

	/** {@inheritDoc} */
	@Override
	public Object getParent(Object element) {
		if(element instanceof EObject) {
			EObject me = (EObject)element;
			if(IPersistencyService.getInstance().isTopLevelElement(me)) {
				// delegate to persistency service
				return IPersistencyService.getInstance();
			}
			return me.eContainer();
		}
		if(element instanceof TutorialStepBase) {
			TutorialStepBase step = (TutorialStepBase)element;
			return step.getParent() == null ? IPersistencyService.getInstance() : step.getParent();
		}
		return null;
	}

	/**
	 * Toggle the expert view. When expert view is active then more objects are
	 * displayed in the tree.
	 */
	public void toggleExpertView() {
		((NavigatorService)INavigatorService.getInstance()).toggleExpertMode();
	}
}
