/*-------------------------------------------------------------------------+
| Copyright 2020 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.internal.views.marker;

import static org.fortiss.tooling.common.ui.javafx.util.JavaFXUtils.createToolTip;
import static org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity.DEBUG;
import static org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity.ERROR;
import static org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity.FATAL;
import static org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity.INFO;
import static org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity.WARNING;

import java.util.HashMap;
import java.util.Map;

import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.common.ui.javafx.layout.CompositeFXControllerBase;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity;
import org.fortiss.tooling.kernel.ui.internal.MarkerService;
import org.fortiss.tooling.kernel.ui.service.IMarkerService;

import javafx.fxml.FXML;
import javafx.scene.Node;
import javafx.scene.control.ToggleButton;
import javafx.scene.image.ImageView;
import javafx.scene.layout.BorderPane;

/**
 * Controller class for the "marker view". References the FXML layout and contains all control
 * logic.
 * 
 * @author abid
 */
public class MarkerViewController extends CompositeFXControllerBase<Node, Node> {

	/** Debug Button */
	@FXML
	private ToggleButton debugBtn;
	/** Info Button */
	@FXML
	private ToggleButton infoBtn;
	/** Warning Button */
	@FXML
	private ToggleButton warningBtn;
	/** Error Button */
	@FXML
	private ToggleButton errorBtn;
	/** Fatal Button */
	@FXML
	private ToggleButton fatalBtn;
	/** Border Pane for the tree table */
	@FXML
	private BorderPane pane;

	/** Toggle State */
	private Map<ESeverity, Boolean> toggleState = new HashMap<ESeverity, Boolean>();
	/** Tree table */
	private DynamicTreeTableViewer<Object> treeTable;

	/** Constructor. */
	@SuppressWarnings("unchecked")
	public MarkerViewController(Map<ESeverity, Boolean> toggleState) {
		this.toggleState = toggleState;
		((MarkerService)IMarkerService.getInstance()).setMarkerViewPart(this);
	}

	/** Called when debug checkbox is pushed. */
	public void onDebugBtn() {
		toggleState.put(DEBUG, debugBtn.isSelected());
		refresh();
	}

	/** Called when info check-box is pushed. */
	public void onInfoBtn() {
		toggleState.put(INFO, infoBtn.isSelected());
		refresh();
	}

	/** Called when warning check-box is pushed. */
	public void onWarningBtn() {
		toggleState.put(WARNING, warningBtn.isSelected());
		refresh();
	}

	/** Called when error check-box is pushed. */
	public void onErrorBtn() {
		toggleState.put(ERROR, errorBtn.isSelected());
		refresh();
	}

	/** Called when fatal check-box is pushed. */
	public void onFatalBtn() {
		toggleState.put(FATAL, fatalBtn.isSelected());
		refresh();
	}

	/** {@inheritDoc} */
	@Override
	public String getFXMLLocation() {
		return "MarkerView.fxml";
	}

	/** Refreshes the tree table. */
	public void refresh() {
		if(treeTable != null) {
			treeTable.update();
		}
	}

	/** {@inheritDoc} */
	@Override
	public void initialize() {
		DynamicTreeContentProviderBase<Object> contentProvider = new MarkerViewContentProvider();
		DynamicTreeTableUIProviderBase<Object> uiProvider = new MarkerViewUIProvider();
		treeTable = new DynamicTreeTableViewer<Object>(toggleState, false, 2, contentProvider,
				uiProvider);
		treeTable.addColumn("Severity", 80);
		treeTable.addColumn("Element", 80);
		treeTable.addColumn("Explanation", 80);
		treeTable.addColumn("Project", 80);

		// set up icons
		ImageView debugImg = IMarkerService.getInstance().getFXImageFor(DEBUG, false);
		debugBtn.setGraphic(debugImg);
		debugBtn.setText("");
		debugBtn.setTooltip(createToolTip(DEBUG.name()));
		toggleState.put(DEBUG, false);

		ImageView errorImg = IMarkerService.getInstance().getFXImageFor(ERROR, false);
		errorBtn.setGraphic(errorImg);
		errorBtn.setText("");
		errorBtn.setTooltip(createToolTip(ERROR.name()));
		toggleState.put(ERROR, true);
		errorBtn.setSelected(true);

		ImageView fatalImg = IMarkerService.getInstance().getFXImageFor(FATAL, false);
		fatalBtn.setGraphic(fatalImg);
		fatalBtn.setText("");
		fatalBtn.setTooltip(createToolTip(FATAL.name()));
		toggleState.put(FATAL, true);
		fatalBtn.setSelected(true);

		ImageView infoImg = IMarkerService.getInstance().getFXImageFor(INFO, false);
		infoBtn.setGraphic(infoImg);
		infoBtn.setText("");
		infoBtn.setTooltip(createToolTip(INFO.name()));
		toggleState.put(INFO, false);

		ImageView warnImg = IMarkerService.getInstance().getFXImageFor(WARNING, false);
		warningBtn.setGraphic(warnImg);
		warningBtn.setText("");
		warningBtn.setTooltip(createToolTip(WARNING.name()));
		toggleState.put(WARNING, true);
		warningBtn.setSelected(true);

		pane.setCenter(treeTable.getControl());
		refresh();
	}
}
