/*-------------------------------------------------------------------------+
| Copyright 2020 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.introspection;

import static org.fortiss.tooling.common.util.LambdaUtils.getFirst;

import java.util.stream.Collectors;

import org.fortiss.tooling.common.ui.javafx.layout.CompositeFXControllerBase;
import org.fortiss.tooling.kernel.introspection.IIntrospectionItem;
import org.fortiss.tooling.kernel.introspection.KernelIntrospectionSystemService;

import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.fxml.FXML;
import javafx.scene.Node;
import javafx.scene.control.TreeCell;
import javafx.scene.control.TreeItem;
import javafx.scene.control.TreeView;
import javafx.scene.layout.BorderPane;

/**
 * Controller for the Java FX-based UI common to all KISS-based views. The specific views to be
 * displayed have extend the abstract {@link KISSCompositeFXController} class.
 * 
 * @author munaro
 */
public class KISSController extends CompositeFXControllerBase<BorderPane, Node>
		implements ChangeListener<TreeItem<IIntrospectionItem>> {

	/** The pane of the view. */
	@FXML
	protected BorderPane borderPane;

	/** The introspection service. */
	protected KernelIntrospectionSystemService kiss;
	/** The tree viewer. */
	private TreeView<IIntrospectionItem> treeview;
	/** The {@link KISSCompositeFXController} of the specific KISS view to be displayed. */
	private KISSCompositeFXController<? extends Node> composite;

	/** Constructor. */
	@SuppressWarnings("unchecked")
	public KISSController(KISSCompositeFXController<? extends Node> composite) {
		super(composite);
		this.composite = composite;
		this.kiss = KernelIntrospectionSystemService.getInstance();
	}

	/**
	 * Implementation for tree cells returning the
	 * {@link IIntrospectionItem#getIntrospectionLabel()}.
	 */
	private class TreeCellImpl extends TreeCell<IIntrospectionItem> {
		/** {@inheritDoc} */
		@Override
		protected void updateItem(IIntrospectionItem item, boolean empty) {
			super.updateItem(item, empty);
			String text = null;
			if(!empty && item != null) {
				text = item.getIntrospectionLabel();
			}
			setText(text);
			setGraphic(null);
		}
	}

	/** {@inheritDoc} */
	@Override
	public String getFXMLLocation() {
		return "KISS.fxml";
	}

	/** {@inheritDoc} */
	@Override
	public void initialize() {
		TreeItem<IIntrospectionItem> root = new TreeItem<IIntrospectionItem>(kiss);
		root.getChildren()
				.addAll(kiss.getIntrospectionItems().stream().map((IIntrospectionItem iii) -> {
					return new TreeItem<IIntrospectionItem>(iii);
				}).collect(Collectors.toList()));
		root.setExpanded(true);
		root.getChildren().sort((c0, c1) -> {
			String l0 = c0.getValue().getIntrospectionLabel();
			String l1 = c1.getValue().getIntrospectionLabel();
			return l0.compareTo(l1);
		});
		treeview = new TreeView<IIntrospectionItem>(root);
		treeview.setCellFactory((TreeView<IIntrospectionItem> tv) -> {
			return new TreeCellImpl();
		});
		treeview.getSelectionModel().selectedItemProperty().addListener(this);

		borderPane.setLeft(treeview);

		Node center = getFirst(getContainments()).get().getLayout();
		borderPane.setCenter(center);
	}

	/** {@inheritDoc} */
	@Override
	public void changed(ObservableValue<? extends TreeItem<IIntrospectionItem>> observable,
			TreeItem<IIntrospectionItem> oldValue, TreeItem<IIntrospectionItem> newValue) {
		composite.changed(newValue.getValue());
	}
}
