/*******************************************************************************
 * Copyright (c) 2011, 2018 fortiss GmbH.
 * 
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 *******************************************************************************/
package org.fortiss.tooling.kernel.ui.introspection.details.handler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.kernel.introspection.items.EObjectAwareKISSDetailsItemBase;

import javafx.scene.Node;
import javafx.scene.control.ContextMenu;

/**
 * Base class for details UI implementations with filtered tree viewer.
 * 
 * @author hoelzl
 */
public abstract class EObjectAwareKISSDetailsUIHandlerBase extends KISSDetailsUIHandlerBase {

	/** Creates the default tree columns. */
	protected final void createDefaultTreeColumns(DynamicTreeTableViewer<Object> viewer,
			String text0, int width0, String text1, int width1, String text2, int width2) {
		viewer.addColumn(text0, width0);
		viewer.addColumn(text1, width1);
		viewer.addColumn(text2, width2);
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeTableUIProviderBase<Object> createUIProvider() {
		return new DynamicTreeTableUIProviderBase<Object>() {
			@Override
			public String getLabel(Object element, int column) {
				if(element instanceof Class) {
					return column == 0 ? ((Class<?>)element).getSimpleName() : "";
				}
				if(element instanceof Pair) {
					Pair<?, ?> pair = (Pair<?, ?>)element;
					if(pair.getLeft() != null) {
						Object provider = pair.getLeft();
						Class<?> regClass = (Class<?>)pair.getRight();
						switch(column) {
							case 0:
								return provider.getClass().getSimpleName();
							case 1:
								return provider.getClass().getName();
							case 2:
								return regClass.getName();
						}
					}
				}
				return "";
			}

			@Override
			public Node getIconNode(Object element, int column) {
				return null;
			}

			/** {@inheritDoc} */
			@Override
			public ContextMenu createContextMenu(Object element, int column) {
				if(element instanceof Pair) {
					Pair<?, ?> pair = (Pair<?, ?>)element;
					if(pair.getLeft() == null) {
						return null;
					}
					Object provider = pair.getLeft();
					Class<?> regClass = (Class<?>)pair.getRight();
					switch(column) {
						case 0:
							break;
						case 1:
							return createCopyClassNameMenuItem(provider.getClass());
						case 2:
							return createCopyClassNameMenuItem(regClass);
						default:
					}
				}
				return null;
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected boolean testObjectFilter(String filterExpression, Object element) {
		Class<?> toMatch = null;
		if(element instanceof Class) {
			toMatch = (Class<?>)element;
		} else if(element instanceof Pair) {
			Pair<?, ?> pair = (Pair<?, ?>)element;
			if(pair.getLeft() != null) {
				toMatch = pair.getLeft().getClass();
			}
		}
		if(toMatch == null) {
			return false;
		}
		filterExpression = filterExpressionSanityCheck(filterExpression);
		return toMatch.getSimpleName().matches(filterExpression);
	}

	/** {@inheritDoc} */
	@Override
	protected Comparator<Object> createSorter() {
		return (o1, o2) -> {
			if(o1 instanceof Pair && o2 instanceof Pair) {
				Pair<?, ?> pair1 = (Pair<?, ?>)o1;
				Pair<?, ?> pair2 = (Pair<?, ?>)o2;
				o1 = pair1.getLeft().getClass();
				o2 = pair2.getLeft().getClass();
				// fall through to next if statement
			}
			if(o1 instanceof Class && o2 instanceof Class) {
				return ((Class<?>)o1).getSimpleName().compareTo(((Class<?>)o2).getSimpleName());
			}
			return 0;
		};
	}

	/** Tree content provider for handler registrations with two classes. */
	protected static abstract class EObjectAwareTreeContentProviderBase
			extends DynamicTreeContentProviderBase<Object> {
		/** Returns the input object for this content provider. */
		protected abstract EObjectAwareKISSDetailsItemBase<?> getInputObject();

		/** {@inheritDoc} */
		@Override
		public Collection<? extends Object> getChildren(Object parentElement) {
			if(parentElement == getInputObject()) {
				return getInputObject().getHandlerKeyClasses();
			}
			if(parentElement instanceof Class<?>) {
				Class<?> regClass = (Class<?>)parentElement;
				List<?> handlerList = getInputObject().getHandlerList(regClass);
				if(handlerList == null || handlerList.isEmpty()) {
					return null;
				}
				Collection<Pair<?, ?>> pairs = new ArrayList<Pair<?, ?>>(handlerList.size());
				for(Object handler : handlerList) {
					pairs.add(new ImmutablePair<Object, Class<?>>(handler, regClass));
				}
				return pairs;
			}
			return null;
		}
	}
}
