/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.introspection.details.handler;

import static java.util.Arrays.asList;

import java.util.Collection;

import org.eclipse.core.resources.IFile;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.kernel.extension.IEclipseResourcePostLoadProvider;
import org.fortiss.tooling.kernel.extension.IEclipseResourceStorageLocationProvider;
import org.fortiss.tooling.kernel.extension.data.ModelStorageError;
import org.fortiss.tooling.kernel.introspection.items.EclipseResourceStorageKISSDetailsItem;

import javafx.scene.Node;
import javafx.scene.control.ContextMenu;

/**
 * Introspection UI handler for the {@link EclipseResourceStorageKISSDetailsItem}s.
 * 
 * @author hoelzl
 */
public final class EclipseResourceStorageKISSDetailsUIHandler extends KISSDetailsUIHandlerBase {

	/** {@inheritDoc} */
	@Override
	public Node createDisplayControl() {
		String heading = "Type to search registered Eclipse resource providers:";
		String footer = "Number of currently registered storage providers: " +
				getRootObject().getStorageProviderList().size() + " (providing " +
				getRootObject().getLoadedFiles().size() + " file[s], " +
				getRootObject().getErrorFiles().size() + " erroneous).";
		return createFilteredTree(heading, footer);
	}

	/** {@inheritDoc} */
	@Override
	protected void createTreeColumns(DynamicTreeTableViewer<Object> tree) {
		tree.addColumn("Category / Provider / File", 450);
		tree.addColumn("Implementation Class / Path / Storage Error Message", 400);
	}

	/** String array of root categories. */
	private final String[] ROOT_CATEGORY =
			new String[] {"Storage Provider", "Loaded Files", "Erroneous Files"};

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeContentProviderBase<Object> createContentProvider() {
		return new DynamicTreeContentProviderBase<Object>() {

			@Override
			public Collection<? extends Object> getChildren(Object parentElement) {
				if(parentElement == getRootObject()) {
					return asList(ROOT_CATEGORY);
				}
				if(ROOT_CATEGORY[0].equals(parentElement)) {
					return getRootObject().getStorageProviderList();
				}
				if(ROOT_CATEGORY[1].equals(parentElement)) {
					return getRootObject().getLoadedFiles();
				}
				if(ROOT_CATEGORY[2].equals(parentElement)) {
					return getRootObject().getErrorFiles();
				}
				return null;
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeTableUIProviderBase<Object> createUIProvider() {
		return new DynamicTreeTableUIProviderBase<Object>() {
			@Override
			public String getLabel(Object parentElement, int columnIndex) {
				if(parentElement instanceof IEclipseResourceStorageLocationProvider ||
						parentElement instanceof IEclipseResourcePostLoadProvider) {
					return columnIndex == 0 ? parentElement.getClass().getSimpleName()
							: parentElement.getClass().getName();
				}
				if(parentElement instanceof String) {
					return columnIndex == 0 ? (String)parentElement : "";
				}
				if(parentElement instanceof IFile) {
					IFile file = (IFile)parentElement;
					return columnIndex == 0 ? file.getName() : file.getFullPath().toString();
				}
				if(parentElement instanceof ModelStorageError) {
					ModelStorageError error = (ModelStorageError)parentElement;
					return columnIndex == 0 ? "" : error.getMessage();
				}
				return "";
			}

			/** {@inheritDoc} */
			@Override
			public ContextMenu createContextMenu(Object element, int column) {
				return createCopyClassNameMenuItem(element.getClass());
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected EclipseResourceStorageKISSDetailsItem getRootObject() {
		return (EclipseResourceStorageKISSDetailsItem)dataItem;
	}
}
