/*******************************************************************************
 * Copyright (c) 2011, 2018 fortiss GmbH.
 * 
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 *******************************************************************************/
package org.fortiss.tooling.kernel.ui.introspection.details.handler;

import static java.util.Comparator.comparing;
import static org.fortiss.tooling.kernel.ui.introspection.ClipboardCopyHelper.createCopyClassNameMenu;

import java.util.Comparator;

import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.kernel.ui.introspection.details.DetailsUIHandlerBase;

import javafx.scene.Node;
import javafx.scene.control.ContextMenu;
import javafx.scene.control.Label;
import javafx.scene.control.TextField;
import javafx.scene.layout.BorderPane;

/**
 * Base class for details UI implementations with filtered tree viewer.
 * 
 * @author hoelzl
 */
public abstract class KISSDetailsUIHandlerBase extends DetailsUIHandlerBase {
	/** {@inheritDoc} */
	@Override
	public Node createDisplayControl() {
		return new Label("No details available!");
	}

	/** Creates the filtered tree control. */
	protected final BorderPane createFilteredTree(String searchFieldAdvice, String footerLabel) {
		BorderPane pane = new BorderPane();
		DynamicTreeTableViewer<Object> treeView = new DynamicTreeTableViewer<Object>(
				getRootObject(), false, 0, createContentProvider(), createUIProvider());
		createTreeColumns(treeView);
		pane.setCenter(treeView.getControl());
		if(searchFieldAdvice != null) {
			TextField searchField = new TextField();
			searchField.setPromptText(searchFieldAdvice);
			searchField.textProperty().addListener((obs, oldValue, newValue) -> {
				treeView.update();
			});
			pane.setTop(searchField);
		}
		if(footerLabel != null) {
			Label footer = new Label(footerLabel);
			pane.setBottom(footer);
		}
		return pane;
	}

	/** Returns the root object for the tree-table viewer. */
	protected abstract Object getRootObject();

	/** Returns the content provider for the tree-table viewer. */
	protected abstract DynamicTreeContentProviderBase<Object> createContentProvider();

	/** Returns the label provider for the tree-table viewer. */
	protected abstract DynamicTreeTableUIProviderBase<Object> createUIProvider();

	/** Returns the Comparator used to sort the elements in the viewer. */
	protected Comparator<Object> createSorter() {
		return comparing(Object::toString);
	}

	/**
	 * Tests the given object's toString representation for inclusion under the given text filter
	 * expression. The filter expression supports wildcard matching with {@code '?'} for a single
	 * character and {@code '*'} for arbitrary many characters.
	 */
	protected boolean testObjectFilter(String filterExpression, Object dataItem) {
		if(dataItem == null) {
			return false;
		}
		filterExpression = filterExpressionSanityCheck(filterExpression);
		return dataItem.toString().matches(filterExpression);
	}

	/** Tests the given filter expression and replaces it if necessary. */
	protected String filterExpressionSanityCheck(String filterExpression) {
		if(filterExpression == null || filterExpression.trim().equals("")) {
			filterExpression = ".*";
		} else {
			if(!filterExpression.endsWith("*")) {
				filterExpression += '*';
			}
			if(!filterExpression.startsWith("*")) {
				filterExpression = '*' + filterExpression;
			}
			filterExpression = filterExpression.replace("?", ".?").replace("*", ".*");
		}
		return filterExpression;
	}

	/** Creates the columns for the tree-table viewer. */
	protected abstract void createTreeColumns(DynamicTreeTableViewer<Object> viewer);

	/** Creates a context menu with an entry to copy the given String. */
	protected final ContextMenu createCopyClassNameMenuItem(Class<?> toCopy) {
		return createCopyClassNameMenu(toCopy);
	}
}
