/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.introspection.details.handler;

import static java.util.Arrays.asList;

import java.util.Collection;

import org.eclipse.emf.ecore.EObject;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.kernel.extension.IStorageProvider;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.extension.data.ModelStorageError;
import org.fortiss.tooling.kernel.introspection.items.PersistencyKISSDetailsItem;
import org.fortiss.tooling.kernel.service.IPersistencyService;
import org.fortiss.tooling.kernel.ui.service.IModelElementHandlerService;

import javafx.scene.Node;
import javafx.scene.control.ContextMenu;

/**
 * Introspection UI handler for the {@link IPersistencyService}.
 * 
 * @author hoelzl
 */
public final class PersistencyKISSDetailsUIHandler extends KISSDetailsUIHandlerBase {

	/** {@inheritDoc} */
	@Override
	public Node createDisplayControl() {
		String heading = "Type to search registered persistency providers:";
		String footer = "Number of currently registered providers: " +
				getRootObject().getStorageProviders().size();
		return createFilteredTree(heading, footer);
	}

	/** {@inheritDoc} */
	@Override
	protected void createTreeColumns(DynamicTreeTableViewer<Object> tree) {
		tree.addColumn("Storage Provider Class / Toplevel Element", 450);
		tree.addColumn("Toplevel EObject Class / Storage Error Message", 400);
		tree.addColumn("Error Cause Exception", 400);
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeContentProviderBase<Object> createContentProvider() {
		return new DynamicTreeContentProviderBase<Object>() {

			@Override
			public Collection<? extends Object> getChildren(Object parentElement) {
				if(parentElement == getRootObject()) {
					return getRootObject().getStorageProviders();
				}
				if(parentElement instanceof IStorageProvider) {
					IStorageProvider prov = (IStorageProvider)parentElement;
					return asList(new Object[] {new ToplevelOrErrorTreeNode(prov, true),
							new ToplevelOrErrorTreeNode(prov, false)});
				}
				if(parentElement instanceof ToplevelOrErrorTreeNode) {
					ToplevelOrErrorTreeNode node = (ToplevelOrErrorTreeNode)parentElement;
					if(node.isToplevel) {
						return node.provider.getTopLevelElements();
					}
					return node.provider.getStorageErrors();
				}
				return null;
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeTableUIProviderBase<Object> createUIProvider() {
		return new DynamicTreeTableUIProviderBase<Object>() {
			@Override
			public ContextMenu createContextMenu(Object element, int column) {
				if(element instanceof IStorageProvider) {
					return createCopyClassNameMenuItem(element.getClass());
				} else if(element instanceof ITopLevelElement) {
					EObject eo = ((ITopLevelElement)element).getRootModelElement();
					return createCopyClassNameMenuItem(eo.getClass());
				}
				return null;
			}

			@Override
			public String getLabel(Object parentElement, int columnIndex) {
				if(parentElement instanceof IStorageProvider) {
					IStorageProvider isp = (IStorageProvider)parentElement;
					return columnIndex == 0 ? isp.getClass().getName() : "";
				}
				if(parentElement instanceof ToplevelOrErrorTreeNode) {
					ToplevelOrErrorTreeNode node = (ToplevelOrErrorTreeNode)parentElement;
					if(node.isToplevel) {
						return columnIndex == 0 ? "Toplevel Elements" : "";
					}
					return columnIndex == 0 ? "Model Storage Errors" : "";
				}
				if(parentElement instanceof ITopLevelElement) {
					ITopLevelElement top = (ITopLevelElement)parentElement;
					switch(columnIndex) {
						case 0:
							return IModelElementHandlerService.getInstance()
									.getName(top.getRootModelElement());
						case 1:
							return top.getRootModelElement().getClass().getName();
						default:
							return "";
					}
				}
				if(parentElement instanceof ModelStorageError) {
					ModelStorageError error = (ModelStorageError)parentElement;
					switch(columnIndex) {
						case 0:
							return "";
						case 1:
							return error.getMessage();
						case 2:
							return error.getCause().toString();
					}
				}
				return "";
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected PersistencyKISSDetailsItem getRootObject() {
		return (PersistencyKISSDetailsItem)dataItem;
	}

	/** Class for tree nodes below storage providers. */
	private static class ToplevelOrErrorTreeNode {
		/** The storage provider. */
		public final IStorageProvider provider;
		/** The top level flag. */
		public final boolean isToplevel;

		/** Constructor. */
		public ToplevelOrErrorTreeNode(IStorageProvider provider, boolean toplevel) {
			this.provider = provider;
			this.isToplevel = toplevel;
		}
	}
}
