/*******************************************************************************
 * Copyright (c) 2011, 2018 fortiss GmbH.
 * 
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
 * which is available at https://www.apache.org/licenses/LICENSE-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
 *******************************************************************************/
package org.fortiss.tooling.kernel.ui.introspection.details.handler;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

import java.util.Collection;

import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeContentProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableUIProviderBase;
import org.fortiss.tooling.common.ui.javafx.control.treetableview.DynamicTreeTableViewer;
import org.fortiss.tooling.kernel.extension.IPrototypeProvider;
import org.fortiss.tooling.kernel.extension.data.Prototype;
import org.fortiss.tooling.kernel.extension.data.PrototypeCategory;
import org.fortiss.tooling.kernel.service.IPrototypeService;

import javafx.scene.Node;
import javafx.scene.control.ContextMenu;

/**
 * Introspection UI handler for the {@link IPrototypeService}.
 * 
 * @author hoelzl
 */
public final class PrototypeKISSDetailsUIHandler extends KISSDetailsUIHandlerBase {

	/** {@inheritDoc} */
	@Override
	public Node createDisplayControl() {
		String footer = "Number of currently registered Prototypes: " +
				IPrototypeService.getInstance().getAllPrototypes().size() + " (" +
				IPrototypeService.getInstance().getPrimaryPrototypes().size() +
				" primary Prototypes)";
		return createFilteredTree(null, footer);
	}

	/** {@inheritDoc} */
	@Override
	protected void createTreeColumns(DynamicTreeTableViewer<Object> viewer) {
		viewer.addColumn("Category / Prototype", 200);
		viewer.addColumn("Provider Class", 400);
		viewer.addColumn("EObject Class", 400);
		viewer.addColumn("Primary", 80);
	}

	/** Returns the label for the given Prototype and column. */
	private String getPrototypeLabelForFirstColumn(Prototype arche, int column) {
		switch(column) {
			case 0:
				return arche.getName();
			case 1:
				return findPrototypeProviderClass(arche).getName();
			case 2:
				return arche.getPrototype().getClass().getName();
			case 3:
				return arche.isPrimary() ? "X" : null;
			default:
		}
		return null;
	}

	/** Find the Prototype provider for the given Prototype. */
	private Class<? extends IPrototypeProvider> findPrototypeProviderClass(Prototype arche) {
		for(IPrototypeProvider provider : IPrototypeService.getInstance().getPrototypeProviders()) {
			if(provider.getPrototypes().contains(arche)) {
				return provider.getClass();
			}
		}
		return null;
	}

	/** {@inheritDoc} */
	@Override
	protected Object getRootObject() {
		return IPrototypeService.getInstance();
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeContentProviderBase<Object> createContentProvider() {
		return new DynamicTreeContentProviderBase<Object>() {
			/** {@inheritDoc} */
			@Override
			protected Collection<? extends Object> getChildren(Object parent) {
				if(parent instanceof IPrototypeService) {
					return ((IPrototypeService)parent).getAllTopLevelPrototypesCategories();
				}
				if(parent instanceof PrototypeCategory) {
					PrototypeCategory cat = (PrototypeCategory)parent;
					return asList(cat.getChildren());
				}
				// Prototype
				return emptyList();
			}
		};
	}

	/** {@inheritDoc} */
	@Override
	protected DynamicTreeTableUIProviderBase<Object> createUIProvider() {
		return new DynamicTreeTableUIProviderBase<Object>() {
			@Override
			public String getLabel(Object element, int column) {
				if(element instanceof Prototype) {
					return getPrototypeLabelForFirstColumn((Prototype)element, column);
				}
				if(element instanceof PrototypeCategory) {
					return column == 0 ? ((PrototypeCategory)element).getName() : null;
				}
				if(element == IPrototypeService.getInstance()) {
					return column == 0 ? "Prototype Service" : null;
				}
				return null;
			}

			@Override
			public Node getIconNode(Object element, int column) {
				return null;
			}

			/** {@inheritDoc} */
			@Override
			public ContextMenu createContextMenu(Object element, int column) {
				if(element instanceof Prototype) {
					Prototype arche = (Prototype)element;
					Class<?> toCopy;
					if(column == 1) {
						toCopy = findPrototypeProviderClass(arche);
					} else if(column == 2) {
						toCopy = arche.getPrototype().getClass();
					} else {
						toCopy = null;
					}
					if(toCopy != null) {
						return createCopyClassNameMenuItem(toCopy);
					}
				}
				return null;
			}
		};
	}
}
