/*-------------------------------------------------------------------------+
| Copyright 2015 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.presentation;

import static org.eclipse.ui.ISharedImages.IMG_DEC_FIELD_ERROR;
import static org.eclipse.ui.ISharedImages.IMG_DEC_FIELD_WARNING;
import static org.eclipse.ui.PlatformUI.getWorkbench;

import java.util.Collection;
import java.util.List;

import org.apache.commons.lang.WordUtils;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DecorationOverlayIcon;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity;
import org.fortiss.tooling.kernel.ui.ToolingKernelUIActivator;
import org.fortiss.tooling.kernel.ui.service.IMarkerService;

/**
 * {@link LabelProvider} for {@link EObject}s, that decorates the icon according to possibly
 * existing {@link IConstraintViolation} for the element.
 * 
 * @author bayha
 */
public class ConstraintViolationAwareLabelProviderBase extends ModelElementLabelProvider {
	/** Text with in characters, for tool tips. */
	protected static final int TOOL_TIP_TEXT_WIDTH = 60;

	/** Time in ms, tool tips (potentially with warnings are displayed). */
	protected static final int TOOL_TIP_DISPLAY_TIME = 100000;

	/** Constructor. */
	public ConstraintViolationAwareLabelProviderBase() {
		super();
	}

	/**
	 * Retrieves all {@link IConstraintViolation}s for 'object', which have
	 * 'object' as source.
	 * Applies {@link #filterViolations(Collection, Object)}.
	 * 
	 * @param object
	 *            object, which caused violation for 'object'.
	 * @return A {@link List} of {@link IConstraintViolation}s.
	 */
	protected Collection<IConstraintViolation<?>> getViolationsForObject(Object object) {
		Collection<IConstraintViolation<?>> violations =
				IMarkerService.getInstance().getViolations((EObject)object);

		return filterViolations(violations, object);
	}

	/**
	 * Filters ConstraintViolations before they are taken into account for decorating the label.
	 * 
	 * This Method might be overwritten, if filtering is intended.
	 * 
	 * @param toFilter
	 *            {@link Collection} of {@link IConstraintViolation}s to be filtered.
	 * @param object
	 *            {@link Object}, for which a label shall be provided.
	 * @return Filtered {@link Collection} of {@link IConstraintViolation}s.
	 */
	protected Collection<IConstraintViolation<?>>
			filterViolations(Collection<IConstraintViolation<?>> toFilter, Object object) {
		return toFilter;
	}

	/**
	 * Retrieves a standard image, to be displayed as icon.
	 * 
	 * @param element
	 *            The {@link Object} to get an image for.
	 * @return {@link ImageDescriptor} for fall back {@link Image}.
	 */
	protected ImageDescriptor getFallbackImage(Object element) {
		return ToolingKernelUIActivator.getImageDescriptor("icons/transparent_icon.gif");
	}

	/**
	 * Retrieves the base image for the icon, which is decorated then with error or warning markers.
	 * 
	 * @param element
	 *            The {@link Object} to get an image for.
	 * @return {@link Image} to be decorated and displayed.
	 */
	protected Image getBaseImage(Object element) {
		Image image = super.getImage(element);

		if(image == null) {
			image = getFallbackImage(element).createImage();
		}

		return image;
	}

	/** {@inheritDoc} */
	@Override
	public Image getImage(Object element) {
		Image image = getBaseImage(element);

		ESeverity highestSeverity = ESeverity.INFO;

		Collection<IConstraintViolation<?>> violationsForElementSelection =
				getViolationsForObject(element);

		// Find highest severity.
		for(IConstraintViolation<? extends EObject> v : violationsForElementSelection) {
			if(v.getSeverity().compareTo(highestSeverity) < 0) {
				highestSeverity = v.getSeverity();
			}
		}

		ImageDescriptor overlay;
		switch(highestSeverity) {
			case ERROR:
				overlay = getWorkbench().getSharedImages().getImageDescriptor(IMG_DEC_FIELD_ERROR);
				break;
			case WARNING:
				overlay =
						getWorkbench().getSharedImages().getImageDescriptor(IMG_DEC_FIELD_WARNING);
				break;
			default:
				return image;
		}

		DecorationOverlayIcon decorated = new DecorationOverlayIcon(image,
				new ImageDescriptor[] {null, null, null, overlay, null}, new Point(16, 16));

		return decorated.createImage();
	}

	/** Wraps long lines after TOOL_TIP_TEXT_WIDTH characters. */
	private String wrap(String s) {
		StringBuilder buffer = new StringBuilder();

		String delim = "";
		for(String line : s.trim().split("\n")) {
			buffer.append(delim);
			delim = "\n";
			buffer.append(WordUtils.wrap(line, TOOL_TIP_TEXT_WIDTH, "\n", true));
		}

		return buffer.toString();
	}

	/**
	 * Get ToolTip text for the given 'element'.
	 * 
	 * This text will be displayed in the ToolTip, before the errors.
	 * 
	 * @param element
	 *            The {@link Object} to get an explanation for.
	 * @return A {@link String}, explaining 'element'.
	 */
	protected String getExplanatoryToolTipText(Object element) {
		return "";
	}

	/** {@inheritDoc} */
	@Override
	public String getToolTipText(Object element) {
		StringBuffer ttBuffer = new StringBuffer(getExplanatoryToolTipText(element));

		Collection<IConstraintViolation<?>> violations = getViolationsForObject(element);

		for(IConstraintViolation<?> v : violations) {
			ttBuffer.append("\n\n");
			ttBuffer.append(v.getSeverity());
			ttBuffer.append(": ");
			ttBuffer.append(v.getExplanation());
		}
		return wrap(ttBuffer.toString());
	}

	/** {@inheritDoc} */
	@Override
	public int getToolTipTimeDisplayed(Object object) {
		return TOOL_TIP_DISPLAY_TIME;
	}
}
