/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.service;

import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.fortiss.tooling.kernel.ui.extension.IContextMenuContributor;
import org.fortiss.tooling.kernel.ui.extension.data.ContextMenuContextProvider;
import org.fortiss.tooling.kernel.ui.internal.ActionService;
import org.fortiss.tooling.kernel.ui.internal.ContextMenuService;

import javafx.scene.control.MenuItem;

/**
 * The context menu service allows registration for model element specific
 * context menus. The context menu consists of a stack of separated sections
 * with the following IDs (from top to bottom):
 * <UL>
 * <LI>{@link #TOP_MOST_MENU_SECTION_ID}
 * <LI>{@link #BEFORE_GLOBAL_MENU_SECTION_ID}
 * <LI>{@link ActionService#GLOBAL_DEFAULT_MENU_SECTION_ID}
 * <LI>{@link #AFTER_GLOBAL_MENU_SECTION_ID}
 * <LI>{@link #REPOSITORY_MENU_SECTION_ID}
 * <LI>{@link IWorkbenchActionConstants#MB_ADDITIONS}
 * <LI>{@link #BOTTOM_MOST_MENU_SECTION_ID}
 * </UL>
 * 
 * If some section has no item in it, no separator is displayed.
 * 
 * @see IContextMenuContributor
 * 
 * @author hoelzl
 */
public interface IContextMenuService {

	/** Top-most menu section ID. */
	public static final String TOP_MOST_MENU_SECTION_ID = "topmost";

	/** Before global section menu section ID. */
	public static final String BEFORE_GLOBAL_MENU_SECTION_ID = "beforeglobal";

	/** After global section menu section ID. */
	public static final String AFTER_GLOBAL_MENU_SECTION_ID = "afterglobal";

	/** Repository menu section separator id. */
	public static final String REPOSITORY_MENU_SECTION_ID = "repository";

	/** Bottom-most menu section ID. */
	public static final String BOTTOM_MOST_MENU_SECTION_ID = "bottommost";

	/** The menu section IDs in order from top to bottom. */
	public static final String[] MENU_SECTION_IDS =
			{TOP_MOST_MENU_SECTION_ID, BEFORE_GLOBAL_MENU_SECTION_ID, AFTER_GLOBAL_MENU_SECTION_ID,
					REPOSITORY_MENU_SECTION_ID, BOTTOM_MOST_MENU_SECTION_ID};

	/** Returns the service instance. */
	public static IContextMenuService getInstance() {
		return ContextMenuService.getInstance();
	}

	/** Creates the default context menu for the given context object. */
	MenuManager createDefaultContextMenu(ContextMenuContextProvider contextProvider);

	/** Registers the given contributor with the kernel. */
	void registerContextMenuContributor(IContextMenuContributor contributor);

	/**
	 * Retrieves JavaFX {@link MenuItem}s for the given elements.
	 * 
	 * @param elements
	 *            A {@link List} of {@link EObject}s to get context menu entries for.
	 * @param contextProvider
	 *            The {@link ContextMenuContextProvider} to be used.
	 * @return A {@link List} of {@link MenuItem}s.
	 */
	public List<MenuItem> getFXMenuItemsForElements(List<EObject> elements,
			ContextMenuContextProvider contextProvider);

	/**
	 * Retrieves JavaFX {@link MenuItem}s for the given element.
	 * 
	 * @param element
	 *            An {@link EObject}s to get context menu entries for.
	 * @param contextProvider
	 *            The {@link ContextMenuContextProvider} to be used.
	 * @return A {@link List} of {@link MenuItem}s.
	 */
	public List<MenuItem> getFXMenuItemsForElement(EObject element,
			ContextMenuContextProvider contextProvider);
}
