/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.service;

import java.util.Collection;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation;
import org.fortiss.tooling.kernel.extension.data.IConstraintViolation.ESeverity;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;
import org.fortiss.tooling.kernel.service.IConstraintCheckerService;
import org.fortiss.tooling.kernel.ui.internal.MarkerService;
import org.fortiss.tooling.kernel.ui.internal.views.marker.MarkerViewPart;

import javafx.scene.image.ImageView;

/**
 * The marker service manages model element specific markers. It uses the
 * {@link IConstraintCheckerService} to check constraints on the model and
 * create the UI feedback from the results.
 * 
 * @author hoelzl
 */
public interface IMarkerService {

	/** Returns view id of the marker view. */
	public static final String MARKER_VIEW = MarkerViewPart.ID;

	/** Returns the service instance. */
	public static IMarkerService getInstance() {
		return MarkerService.getInstance();
	}

	/** Returns the constraint violations for the given element. */
	Collection<IConstraintViolation<? extends EObject>> getViolations(EObject element);

	/**
	 * Returns the highest severity for the given element or <code>null</code> if there are no
	 * violations for this element.
	 */
	ESeverity getHighestViolationSeverity(EObject element);

	/**
	 * Refreshes the markers of the given top-level element (and its children
	 * recursively).
	 */
	void refreshMarkers(ITopLevelElement element);

	/** Returns all violations across projects with the given severity. */
	Collection<IConstraintViolation<? extends EObject>>
			getAllViolationsWithSeverity(ESeverity severity);

	/**
	 * Returns the icon appropriate for the severity. Flag indicates small sized
	 * image.
	 */
	ImageDescriptor getImageFor(ESeverity severity, boolean smallDecoration);

	/** Similar to {@link #getImageFor(ESeverity, boolean)} but for Java FX. */
	ImageView getFXImageFor(ESeverity severity, boolean smallDecoration);

	/**
	 * <p>
	 * Returns a shared icon appropriate for the severity. Flag indicates small sized image.
	 * </p>
	 * <p>
	 * Note: the returned image must not be disposed.
	 * </p>
	 */
	Image getSharedImageFor(ESeverity severity, boolean smallDecoration);
}
