/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import static java.lang.String.join;
import static org.conqat.ide.commons.ui.dialog.MessageUtils.showWarning;
import static org.eclipse.jface.dialogs.MessageDialog.openError;
import static org.fortiss.tooling.kernel.ui.util.SelectionUtils.setSelection;
import static org.fortiss.tooling.kernel.utils.EcoreUtils.pickInstanceOf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.fortiss.tooling.kernel.clipboard.ClipboardObjectWithResource;
import org.fortiss.tooling.kernel.extension.data.IElementCompositionContext;
import org.fortiss.tooling.kernel.model.INamedElement;
import org.fortiss.tooling.kernel.model.ISpeciallyCopyiable;
import org.fortiss.tooling.kernel.service.IElementCompositorService;
import org.fortiss.tooling.kernel.ui.dnd.CompositionServiceLocalCopyPasteTransfer;
import org.fortiss.tooling.kernel.ui.internal.editor.ExtendableMultiPageEditor;
import org.fortiss.tooling.kernel.utils.CompositionUtils.PasteContext;

/**
 * Methods for copy/paste transfer. EMF serialization and {@link TextTransfer} is used to transfer
 * an {@link EObject} to and from the clipboard.
 * 
 * @author hoelzl
 * @author hummel
 */
public class CopyPasteUtils {

	/**
	 * Copies the given selected objects to the clipboard.
	 * 
	 * @param sel
	 *            the collection of {@link EObject}s to be copied to the
	 *            clipboard.
	 */
	public static void copyToClipboard(Collection<EObject> sel) {
		try {
			ClipboardObjectWithResource[] content = new ClipboardObjectWithResource[sel.size()];
			int i = 0;
			EcoreUtil.Copier copier = new EcoreUtil.Copier(true, true);
			for(EObject obj : sel) {
				EObject copiedElement = copier.copy(obj);
				ClipboardObjectWithResource copyForClipboard =
						new ClipboardObjectWithResource(copiedElement, obj.eResource());
				content[i] = copyForClipboard;
				i++;
			}
			copier.copyReferences();
			EList<ISpeciallyCopyiable> specialSel =
					pickInstanceOf(ISpeciallyCopyiable.class, new BasicEList<EObject>(sel));
			for(ISpeciallyCopyiable obj : specialSel) {
				obj.specialCopyHook(copier);
			}
			Clipboard clipboard = new Clipboard(Display.getDefault());
			clipboard.setContents(new Object[] {content},
					new Transfer[] {CompositionServiceLocalCopyPasteTransfer.getInstance()},
					DND.CLIPBOARD | DND.SELECTION_CLIPBOARD);
			clipboard.dispose();
		} catch(Exception e) {
			e.printStackTrace();
			openError(Display.getDefault().getActiveShell(), "Copy",
					"Clipboard contents not writeable: " + e.getMessage());
			return;
		}
	}

	/**
	 * Returns elements read from clipboard as container elements
	 * {@link ClipboardObjectWithResource}.
	 */
	private static ClipboardObjectWithResource[] getClipBoardContent() {
		Clipboard clipboard = new Clipboard(Display.getDefault());

		ClipboardObjectWithResource[] content = (ClipboardObjectWithResource[])clipboard
				.getContents(CompositionServiceLocalCopyPasteTransfer.getInstance());
		clipboard.dispose();
		return content;
	}

	/** Returns elements read from clipboard as single {@link EObject}s. */
	private static EObject[] getClipBoardContentElements() {
		ClipboardObjectWithResource[] rawContent = getClipBoardContent();
		if(rawContent != null) {
			EObject[] elementContent =
					Arrays.stream(rawContent).map(obj -> obj.getElement()).toArray(EObject[]::new);
			return elementContent;
		}
		return null;
	}

	/** Returns the stored original {@link Resource} of clipboard's elements. */
	private static Resource getResourceOfClipBoardContent() {
		ClipboardObjectWithResource[] rawContent = getClipBoardContent();
		if(rawContent != null && rawContent.length > 0) {
			// [0]: rawContent has at least one element
			return rawContent[0].getResource();

		}

		return null;
	}

	/**
	 * Attempts to paste the contents of the clipboard into the given target
	 * object. The given element composition context is forwarded to the element
	 * composition service.
	 * 
	 * @param target
	 *            the target element to be passed to the compositor
	 * @param context
	 *            the composition context to be passed to the compositor
	 */
	public static void pasteFromClipboard(EObject target, IElementCompositionContext context) {
		if(!canPasteInto(target, context)) {
			return;
		}

		// canPasteInto() ensures that clip-board is non-empty
		boolean sameResource = getResourceOfClipBoardContent() == target.eResource();
		if(!sameResource) {
			List<String> textLines = new ArrayList<String>();
			textLines.add("You are pasting content into a new resource/project.");
			textLines.add(
					"If you have external references (e.g. references to complex data types, functions or allocations) in this copied content, please know the following:\n");
			textLines.add(
					"It is currently not possible to automatically take all referenced elements by the copied content to the new resource and update the outdated reference links.");
			textLines.add(
					"Therefore, it is possible that your copied content is now missing some references.\n");
			textLines.add(
					"If your content has such external references, you need to manually add them (ports might still have the correct name of the data type but the saved link to the definition does not exist in the background anymore).");
			showWarning("Possible missing references", join("\n", textLines));
			// TODO #4208: This warning is too generic and not user-friendly. The copy&paste
			// functionality should automatically handle these external references, too.
		}

		// Use original (non-copied) references only when copying within the same model resource.
		// Otherwise, the copy in the destination resource will have a reference to the original
		// resource.
		EcoreUtil.Copier copier = new EcoreUtil.Copier(true, sameResource);
		BasicEList<EObject> copiedObjects = new BasicEList<EObject>();
		for(EObject obj : getClipBoardContentElements()) {
			EObject copy = copier.copy(obj);
			adaptCopyNames(copy, target);
			IElementCompositorService.getInstance().compose(target, copy,
					new PasteContext(getEditedEObject(), copier));
			copiedObjects.add(copy);
		}
		copier.copyReferences();

		for(EObject obj : getClipBoardContentElements()) {
			if(obj instanceof ISpeciallyCopyiable) {
				((ISpeciallyCopyiable)obj).specialCopyHook(copier);
			}
		}
		// Replacing the clipboard with the copied objects. The offset is expected to change
		// recursively for multiple pasting of the same object. Following replaces the objects in
		// the clipboard after being offset.
		copyToClipboard(copiedObjects);

		setSelection(copiedObjects);
	}

	/**
	 * Returns whether the current clipboard element is pasteable into the given
	 * target using the given composition context.
	 * 
	 * @param target
	 *            the target element to be passed to the compositor
	 * @param context
	 *            the composition context to be passed to the compositor
	 * @return true if the clipboard element can be composed with the given target
	 */
	public static boolean canPasteInto(EObject target, IElementCompositionContext context) {

		EObject[] contents = getClipBoardContentElements();
		if(contents == null) {
			return false;
		}

		for(EObject insertObj : contents) {
			if(insertObj instanceof ISpeciallyCopyiable) {

				EObject obj = getEditedEObject();
				if(obj != null) {
					if(!((ISpeciallyCopyiable)insertObj).canPaste(obj, target)) {
						return false;
					}
				}
			}
			if(IElementCompositorService.getInstance().canCompose(target, insertObj, context)) {
				return true;
			}
		}
		return false;
	}

	/** Retrieves the object being currently edited, if it can. */
	public static EObject getEditedEObject() {
		IWorkbenchPart activePart = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
				.getActivePage().getActivePart();

		if(activePart instanceof ExtendableMultiPageEditor) {
			return ((ExtendableMultiPageEditor)activePart).getEditedObject();
		}
		return null;
	}

	/** Checks the names of all {@link INamedElement}s in the target and adapts the copy's name. */
	private static void adaptCopyNames(EObject copy, EObject target) {
		if(copy instanceof INamedElement) {
			String copyName = ((INamedElement)copy).getName();
			int iterator = 0;
			EList<EObject> contentsAtTwoLevels = getContentsAtTwoLevels(target);
			for(EObject o : contentsAtTwoLevels) {
				// check class types
				if(o.getClass().isInstance(copy) && o instanceof INamedElement) {
					INamedElement contained = (INamedElement)o;
					String containedName = contained.getName();
					// check name and copy count
					String pattern2 = "Copy of.*";
					if(containedName != null) {
						if(containedName.equals(copyName)) {
							iterator++;
						} else if(copyName.matches(pattern2) && containedName.matches(pattern2)) {
							if(copyName.split(" ")[2].equals(containedName.split(" ")[2])) {
								iterator++;
							}
						} else if(containedName.matches(pattern2)) {
							if(copyName.equals(containedName.split(" ")[2])) {
								iterator++;
							}
						} else if(copyName.matches(pattern2)) {
							if(containedName.equals(copyName.split(" ")[2])) {
								iterator++;
							}
						}
					}
				}
			}

			String prefix = "Copy of ";
			String newName = copyName;
			if(iterator == 1) {
				newName = prefix + copyName;
			} else if(iterator > 1) {
				String realName = copyName.startsWith(prefix) ? copyName.split(" ")[2] : copyName;
				newName = prefix + realName + " (" + iterator + ")";
			}
			((INamedElement)copy).setName(newName);
		}
	}

	/**
	 * Creates a @{link {@link EList} of the eContents of the current {@link EObject} and the
	 * eContents of its container.
	 */
	private static EList<EObject> getContentsAtTwoLevels(EObject target) {
		EList<EObject> allContents = new BasicEList<EObject>();

		if(target.eContainer() != null) {
			allContents.addAll(target.eContainer().eContents());
		}
		allContents.addAll(target.eContents());

		return allContents;
	}
}
