/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import org.conqat.ide.commons.ui.databinding.JFaceObservables;
import org.eclipse.core.databinding.Binding;
import org.eclipse.core.databinding.DataBindingContext;
import org.eclipse.core.databinding.UpdateValueStrategy;
import org.eclipse.core.databinding.conversion.IConverter;
import org.eclipse.core.databinding.observable.value.IObservableValue;
import org.eclipse.core.databinding.validation.IValidator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.jface.databinding.swt.ISWTObservableValue;
import org.eclipse.jface.databinding.swt.typed.WidgetProperties;
import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Widget;

/**
 * Utility methods for data binding support.
 * 
 * @author hoelzl
 */
public final class DataBindingUtils {

	/** Decoration key for edit feedback. */
	public static final String DECORATION_KEY = ControlDecoration.class.getName();

	/**
	 * Bind the {@link Control} and the {@link EObject} together with the given
	 * {@link EStructuralFeature}.
	 * 
	 * @param dbc
	 *            the data binding context to be used
	 * @param control
	 *            the SWT control
	 * @param modelElement
	 *            the model element to be observed
	 * @param feature
	 *            the element's feature to be observed
	 * @return the array of constructed bindings
	 */
	public static Binding[] bind(DataBindingContext dbc, Control control, EObject modelElement,
			EStructuralFeature feature) {
		return performComplexTextBinding(dbc, control,
				ObservableUtils.observeValue(modelElement, feature), null, null, null, null);
	}

	/**
	 * Bind the {@link Control} and the {@link EObject} together with the given
	 * {@link EStructuralFeature} by using the given {@link IConverter}s.
	 * 
	 * @param dbc
	 *            the data binding context to be used
	 * @param control
	 *            the SWT control
	 * @param modelElement
	 *            the model element to be observed
	 * @param feature
	 *            the element's feature to be observed
	 * @param modelToTextConverter
	 *            the converter from the element to the control's text
	 * @param textToModelConverter
	 *            the converter from the control's text to the element
	 */
	public static void bind(DataBindingContext dbc, Control control, EObject modelElement,
			EStructuralFeature feature, IConverter modelToTextConverter,
			IConverter textToModelConverter) {
		performComplexTextBinding(dbc, control, ObservableUtils.observeValue(modelElement, feature),
				modelToTextConverter, textToModelConverter, null, null);
	}

	/**
	 * Performs a complex binding of a text control to a model element. The
	 * validation is performed on modification (i.e. always), while model
	 * updates are only performed on focus out (if validation works). There is
	 * no validation support from model to text, as the model is considered to
	 * be always consistent. If the control has a {@link ControlDecoration} stored under the
	 * {@link #DECORATION_KEY} key, this is used to visualize
	 * the validation result.
	 * 
	 * @param dbc
	 *            the data binding context to be used
	 * @param control
	 *            the SWT control
	 * @param modelValue
	 *            the model element value to be observed
	 * @param modelToTextConverter
	 *            the converter from the element to the control's text
	 * @param textToModelConverter
	 *            the converter from the control's text to the element
	 * @param textValidator
	 *            the validator for the control's text
	 * @param textPostConvertValidator
	 *            the validator for the text after conversion
	 * @return the array of constructed bindings
	 */
	public static Binding[] performComplexTextBinding(DataBindingContext dbc, Control control,
			IObservableValue<?> modelValue, IConverter modelToTextConverter,
			IConverter textToModelConverter, IValidator textValidator,
			IValidator textPostConvertValidator) {

		return performComplexTextBinding(dbc, control, modelValue, null, modelToTextConverter,
				textToModelConverter, textValidator, textPostConvertValidator);
	}

	/**
	 * Performs a complex binding of a text control to a model element. The
	 * validation is performed on modification (i.e. always), while model
	 * updates are only performed on focus out (if validation works). There is
	 * no validation support from model to text, as the model is considered to
	 * be always consistent. If the control has a {@link ControlDecoration} stored under the
	 * {@link #DECORATION_KEY} key, this is used to visualize
	 * the validation result.
	 * 
	 * @param dbc
	 *            the data binding context to be used
	 * @param control
	 *            the SWT control
	 * @param modelValue
	 *            the model element value to be observed
	 * @param modelValueComment
	 *            the model element comment value to be observed
	 * @param modelToTextConverter
	 *            the converter from the element to the control's text
	 * @param textToModelConverter
	 *            the converter from the control's text to the element
	 * @param textValidator
	 *            the validator for the control's text
	 * @param textPostConvertValidator
	 *            the validator for the text after conversion
	 * @return the array of constructed bindings
	 */

	public static Binding[] performComplexTextBinding(DataBindingContext dbc, Control control,
			IObservableValue<?> modelValue, IObservableValue<?> modelValueComment,
			IConverter modelToTextConverter, IConverter textToModelConverter,
			IValidator textValidator, IValidator textPostConvertValidator) {

		Binding[] bindings;

		if(modelValueComment == null) {
			bindings = new Binding[3];
		} else {
			bindings = new Binding[4];
		}

		// normal model to control
		UpdateValueStrategy modelToTextStrategy =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_UPDATE)
						.setConverter(modelToTextConverter);

		// use POLICY_UPDATE to actually perform the update
		UpdateValueStrategy textToModelStrategyWithUpdate =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_UPDATE)
						.setConverter(textToModelConverter).setAfterGetValidator(textValidator)
						.setAfterConvertValidator(textPostConvertValidator);

		// add the "normal" binding to actually update the model
		bindings[0] = dbc.bindValue(observeText(control, SWT.Modify), modelValue,
				textToModelStrategyWithUpdate, modelToTextStrategy);

		// here it is important to only use POLICY_CONVERT, to not write through
		// to the model
		UpdateValueStrategy textToModelStrategyNoUpdate =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_CONVERT)
						.setConverter(textToModelConverter).setAfterGetValidator(textValidator)
						.setAfterConvertValidator(textPostConvertValidator);

		// perform a binding which only serves for validation purposes
		bindings[1] = dbc.bindValue(observeText(control, SWT.Modify), modelValue,
				textToModelStrategyNoUpdate,
				new UpdateValueStrategy(UpdateValueStrategy.POLICY_NEVER));

		// add validation of required/supported
		final Object data = control.getData(DECORATION_KEY);
		if(data instanceof ControlDecoration) {
			bindings[2] = dbc.bindValue(
					JFaceObservables.observeControlDecoration((ControlDecoration)data),
					bindings[1].getValidationStatus(), null, null);
		}

		// add the "normal" binding to actually update the comments
		if(modelValueComment != null) {
			bindings[3] =
					dbc.bindValue(observeText(control, SWT.Modify), modelValueComment, null, null);
		}

		return bindings;
	}

	/**
	 * Returns an {@link ISWTObservableValue} observing this value property on
	 * the given {@link Control}.
	 *
	 * @param event
	 *            the SWT event type to register for change events. May be
	 *            {@link SWT#None}, {@link SWT#Modify}, {@link SWT#FocusOut} or
	 *            {@link SWT#DefaultSelection}.
	 * @param control
	 *            the source {@link Control}
	 * @return an observable value observing this value property on the given
	 *         {@link Control}.
	 */
	public static ISWTObservableValue observeText(Control control, final int event) {
		return WidgetProperties.text(event).observe(control);
	}

	/**
	 * Returns an {@link ISWTObservableValue} observing this value property on
	 * the given {@link Widget}.
	 *
	 * @param widget
	 *            the {@link Widget}
	 * @return an observable value observing this value property on the given
	 *         {@link Widget}.
	 */
	public static ISWTObservableValue observeSelection(Widget widget) {
		return WidgetProperties.widgetSelection().observe(widget);
	}

	/**
	 * Returns an observable value tracking the visible state of the given
	 * {@link Control}.
	 *
	 * @param control
	 *            the {@link Control}
	 * @return an observable value tracking the visible state of the given
	 *         {@link Control}.
	 */
	public static ISWTObservableValue observeVisible(Control control) {
		return WidgetProperties.visible().observe(control);
	}

	/**
	 * Performs a complex binding of a cell editor control to a model element. The
	 * validation is performed on modification (i.e. always), while model
	 * updates are only performed on focus out (if validation works). There is
	 * no validation support from model to text, as the model is considered to
	 * be always consistent. If the control has a {@link ControlDecoration} stored under the
	 * {@link #DECORATION_KEY} key, this is used to visualize
	 * the validation result.
	 * 
	 * @param dbc
	 *            the data binding context to be used
	 * @param control
	 *            the SWT control
	 * @param cellValue
	 *            the cell model element value to be observed
	 * @param modalToCellConverter
	 *            the converter from the cell model element to the control's text
	 * @param cellToModelConverter
	 *            the converter from the control's text to the cell model element
	 * @param validator
	 *            the validate function for the control's text
	 * @return the array of constructed bindings
	 */
	public static Binding[] performCellTextBinding(DataBindingContext dbc, Control control,
			IObservableValue<?> cellValue, IConverter modalToCellConverter,
			IConverter cellToModelConverter, IValidator validator) {

		Binding[] bindings = new Binding[2];

		// normal model to control
		UpdateValueStrategy modelToCellStrategy =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_UPDATE)
						.setConverter(modalToCellConverter);

		// use POLICY_UPDATE to actually perform the update
		UpdateValueStrategy cellToModelStrategyWithUpdate =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_UPDATE)
						.setConverter(cellToModelConverter).setAfterGetValidator(validator);

		// add the "normal" binding to actually update the model
		bindings[0] = dbc.bindValue(observeText(control, SWT.FocusOut), cellValue,
				cellToModelStrategyWithUpdate, modelToCellStrategy);

		// here it is important to only use POLICY_CONVERT, to not write through
		// to the model
		UpdateValueStrategy cellToModelStrategyNoUpdate =
				new UpdateValueStrategy(true, UpdateValueStrategy.POLICY_CONVERT)
						.setConverter(cellToModelConverter).setAfterGetValidator(validator)
						.setAfterConvertValidator(validator);

		// perform a binding which only serves for validation purposes
		bindings[1] = dbc.bindValue(observeText(control, SWT.Modify), cellValue,
				cellToModelStrategyNoUpdate,
				new UpdateValueStrategy(UpdateValueStrategy.POLICY_NEVER));

		return bindings;
	}
}
