/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.ISelectionService;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * Utility methods for {@link EObject} selections.
 * 
 * @author hoelzl
 */
public final class EObjectSelectionUtils {

	/**
	 * Returns the first selected element if it is an EObject or <code>null</code> otherwise.
	 * 
	 * @return the first selected {@link EObject}
	 */
	public static EObject getCurrentSelectionFirstElement() {
		return getFirstElement(getCurrentSelection());
	}

	/**
	 * Returns the list of selected elements filtered for EObjects.
	 * 
	 * @return the filtered list of selected elements
	 */
	public static List<EObject> getCurrentSelectionEObjects() {
		return getEObjectElements(getCurrentSelection());
	}

	/**
	 * Returns the first selected EObject or <code>null</code> if no EObject is
	 * selected. If the selection implements {@link IAdaptable}, it is asked to
	 * adapt to {@link EObject}.
	 * 
	 * @param selection
	 *            the current selection to be search for an {@link EObject}
	 * @return the first {@link EObject} element in the selection
	 */
	public static EObject getFirstElement(ISelection selection) {
		if(selection instanceof IStructuredSelection && !selection.isEmpty()) {
			return eObjectOfObject(((IStructuredSelection)selection).getFirstElement());
		}
		return null;
	}

	/** Try to retrieve an EObject from an Object. */
	private static EObject eObjectOfObject(Object o) {
		if(o instanceof EObject) {
			return (EObject)o;
		}
		if(o instanceof IAdaptable) {
			return ((IAdaptable)o).getAdapter(EObject.class);
		}
		return null;
	}

	/**
	 * Returns the current selection obtained from the platform.
	 * 
	 * @return the current selection
	 */
	public static ISelection getCurrentSelection() {
		ISelectionService selectionService = getSelectionService();
		return selectionService == null ? null : selectionService.getSelection();
	}

	/**
	 * Returns the platform selection service.
	 * 
	 * @return the platform selection service
	 */
	public static ISelectionService getSelectionService() {
		IWorkbenchWindow activeWorkbenchWindow =
				PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		return activeWorkbenchWindow == null ? null : activeWorkbenchWindow.getSelectionService();
	}

	/**
	 * Extracts all {@link EObject}s from the given selection.
	 * 
	 * @param selection
	 *            the selection to be searched for {@link EObject}s
	 * @return the list of selected {@link EObject}s
	 */
	public static List<EObject> getEObjectElements(ISelection selection) {
		List<EObject> result = new ArrayList<EObject>();
		if(selection instanceof StructuredSelection && !selection.isEmpty()) {
			for(Object o : ((StructuredSelection)selection).toList()) {
				EObject eo = eObjectOfObject(o);
				if(eo != null) {
					result.add(eo);
				} else if(o instanceof IAdaptable) {
					EObject eo2 = ((IAdaptable)o).getAdapter(EObject.class);
					if(eo2 != null) {
						result.add(eo2);
					}
				}
			}
		}
		return result;
	}
}
