/*-------------------------------------------------------------------------+
| Copyright 2012 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import static org.conqat.ide.commons.ui.dialog.MessageUtils.askQuestion;
import static org.conqat.ide.commons.ui.dialog.MessageUtils.showError;
import static org.conqat.ide.commons.ui.dialog.MessageUtils.showInfo;
import static org.conqat.ide.commons.ui.dialog.MessageUtils.showWarning;

import java.io.IOException;

import org.conqat.ide.commons.ui.dialog.MessageUtils;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.swt.widgets.Display;
import org.fortiss.tooling.kernel.extension.data.ITopLevelElement;

/**
 * This class contains additional methods w.r.t. {@link MessageUtils}.
 * 
 * @author bugariu
 */
public class MessageUtilsExtended {

	/** Temporary result for the asked question. */
	private static boolean tempQuestionResult;

	/** Asks question in the UI thread (and block). */
	public synchronized static boolean askQuestionInUIThread(final String title,
			final String message) {
		syncExec(() -> tempQuestionResult = askQuestion(title, message));
		return tempQuestionResult;
	}

	/** Get configuration in the UI thread (and block). */
	public synchronized static boolean confirmInUIThread(final String title, final String message) {
		syncExec(() -> tempQuestionResult = MessageUtils.confirm(title, message));
		return tempQuestionResult;
	}

	/** Opens a error dialog in the UI thread (and block). */
	public static void showErrorInUIThread(final String title, final String message) {
		syncExec(() -> showError(title, message));
	}

	/** Opens a warning dialog in the UI thread (and block). */
	public static void showWarningInUIThread(final String title, final String message) {
		syncExec(() -> showWarning(title, message));
	}

	/** Opens an info dialog in the UI thread (and block). */
	public static void showInfoInUIThread(final String title, final String message) {
		syncExec(() -> showInfo(title, message));
	}

	/** Execute runnable in the UI thread */
	private static void syncExec(Runnable runnable) {
		Display.getDefault().syncExec(runnable);
	}

	/**
	 * Opens a dialog to confirm if a dirty {@link ITopLevelElement} should be saved (no-op if the
	 * {@link ITopLevelElement} is not dirty).
	 * 
	 * @return {@code true} if the model is not dirty (either in the first place, or not anymore,
	 *         after successfully saving).
	 */
	public static boolean askSaveDirtyTopLevelElement(ITopLevelElement topelement) {

		if(!topelement.isDirty()) {
			return true;
		}
		if(!confirmInUIThread("Save required",
				"For the selected action, the model may not contain unsaved changes.\n\nOK to save and continue?")) {
			return false;
		}

		try {
			topelement.doSave(new NullProgressMonitor());
		} catch(CoreException | IOException e) {
			showErrorInUIThread("Failed to save project \"" + topelement.getSaveableName(),
					e.getMessage());
			return false;
		}
		return true;
	}
}
