/*-------------------------------------------------------------------------+
| Copyright 2011 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import static java.util.Collections.emptyList;
import static org.conqat.ide.commons.ui.ui.WorkbenchUtils.getActiveEditor;
import static org.fortiss.tooling.common.util.LambdaUtils.filterByType;

import java.util.Collection;
import java.util.List;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.fortiss.tooling.kernel.ui.service.INavigatorService;

/**
 * Utility methods for dealing with selections.
 * 
 * This class contains slight improvements with respect to
 * {@link org.conqat.ide.commons.ui.selection.SelectionUtils}.
 * 
 * @author ratiu
 */
public class SelectionUtils {

	/**
	 * For a selection provider object get the selected object, for simple
	 * selections a cast is performed to the desired type; for a structured
	 * selection returns the first element or <code>null</code> if the selection
	 * is empty. This returns <code>null</code> if the provided selection is
	 * null or the element is not of the required type.
	 * 
	 * @param selectionProvider
	 *            the selection provider
	 * @param type
	 *            the type of elements to be returned
	 * @return first element of the given type
	 */
	public static <T> T checkAndPickFirst(ISelectionProvider selectionProvider, Class<T> type) {
		return checkAndPickFirst(selectionProvider.getSelection(), type);
	}

	/**
	 * For simple selections a cast is performed to the desired type. For a
	 * structured selection this returns the first element or <code>null</code> if the selection is
	 * empty. This also returns <code>null</code> if the
	 * provided selection is <code>null</code> or the element is not of the
	 * required type.
	 * 
	 * @param selection
	 *            the current selection
	 * @param type
	 *            the type of elements to be returned
	 * @return the first element of the given type
	 */
	@SuppressWarnings("unchecked")
	public static <T> T checkAndPickFirst(ISelection selection, Class<T> type) {
		Object selectedElement;

		if(selection == null || selection.isEmpty())
			return null;

		if(!(selection instanceof IStructuredSelection)) {
			selectedElement = selection;
		} else {
			IStructuredSelection structuredSelection = (IStructuredSelection)selection;
			selectedElement = structuredSelection.getFirstElement();
		}

		if(!type.isInstance(selectedElement)) {
			return null;
		}
		return (T)selectedElement;
	}

	/**
	 * Returns a {@link Collection} of all the selected elements that have the given type (possibly
	 * an empty list, in particular if {@code selection} is {@code null} or is not an
	 * {@link IStructuredSelection}.
	 * 
	 * @param selection
	 *            the current selection
	 * @param type
	 *            the type of elements to be returned
	 * @return the {@link Collection} of elements of the given type
	 */
	@SuppressWarnings("unchecked")
	private static <T> Collection<T> selectAll(ISelection selection, Class<T> type) {
		if(selection == null || !(selection instanceof IStructuredSelection)) {
			return emptyList();
		}

		IStructuredSelection structuredSelection = (IStructuredSelection)selection;
		return filterByType(structuredSelection.toList(), type);
	}

	/**
	 * Programmatically sets the objects to be selected in the currently active editor (NOP if there
	 * is none) and the model navigator.
	 * 
	 * @param eObjects
	 *            {@link EObject}s to be selected.
	 */
	public static void setSelection(List<EObject> eObjects) {
		setSelection(eObjects, true, true);
	}

	/**
	 * Programmatically sets the objects to be selected in the currently active editor (NOP if there
	 * is none) and the model navigator.
	 * 
	 * @param eObjects
	 *            {@link EObject}s to be selected.
	 * @param updateNavigator
	 *            Flag whether to update the model navigator
	 * @param updateActiveEditor
	 *            Flag whether to update the currently active editor.
	 */
	public static void setSelection(List<EObject> eObjects, boolean updateNavigator,
			boolean updateActiveEditor) {

		ISelection selection = new StructuredSelection(eObjects);
		if(updateNavigator) {
			INavigatorService ns = INavigatorService.getInstance();
			if(!selectAll(ns.getCurrentSelection(), Object.class).containsAll(eObjects)) {
				ns.setCurrentSelection(selection);
			}
		}

		if(!updateActiveEditor) {
			return;
		}
		IEditorPart activeEditor = getActiveEditor();
		if(activeEditor == null) {
			return;
		}

		IEditorSite editorSite = activeEditor.getEditorSite();
		if(editorSite == null) {
			return;
		}

		ISelectionProvider selectionProvider = editorSite.getSelectionProvider();
		if(selectionProvider == null) {
			return;
		}

		if(!selectAll(selectionProvider.getSelection(), Object.class).containsAll(eObjects)) {
			selectionProvider.setSelection(selection);
		}
	}
}
