/*-------------------------------------------------------------------------+
| Copyright 2016 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import java.util.Map;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.fortiss.tooling.kernel.extension.ITutorialProvider;
import org.fortiss.tooling.kernel.extension.data.Prototype;
import org.fortiss.tooling.kernel.service.ITutorialService;
import org.fortiss.tooling.kernel.ui.ESharedImages;
import org.fortiss.tooling.kernel.ui.service.IActionService;

/**
 * Utility class providing common methods used when implementing UI parts of tutorials.
 * 
 * @author hoelzl
 */
public class TutorialUIServiceUtils {
	/**
	 * Returns whether the given contribution is a new menu prototype instance of the given class.
	 */
	public static boolean isNewMenuContributionPrototype(IContributionItem contribution,
			Class<? extends EObject> clazz) {
		Prototype proto =
				IActionService.getInstance().getNewMenuContributionPrototype(contribution);
		return proto != null && clazz.isInstance(proto.getPrototype());
	}

	/** Creates the tutorial menus with the registered categories and providers. */
	public static MenuManager createTutorialMenu() {
		MenuManager tutorialMenu = new MenuManager("&Tutorial", "tutorial");
		ImageDescriptor imgDescr = ESharedImages.TUTORIAL.getImageDescriptor();
		tutorialMenu.add(new Action("Never shown!", imgDescr) {
			// nothing to do here
		});
		tutorialMenu.setRemoveAllWhenShown(true);
		tutorialMenu.addMenuListener(new IMenuListener() {

			@Override
			public void menuAboutToShow(IMenuManager manager) {
				boolean tutorialRunning = ITutorialService.getInstance().isTutorialActive();
				if(tutorialRunning) {
					createStopTutorialItem(manager);
				} else {
					createTutorialItems(imgDescr, manager);
				}
			}

			/** Creates menu items for registered tutorials. */
			private void createTutorialItems(final ImageDescriptor imgDescr, IMenuManager manager) {
				ITutorialService service = ITutorialService.getInstance();
				for(String cat : service.getProviderCategories()) {
					MenuManager catManager = new MenuManager(cat);
					Map<String, Class<? extends ITutorialProvider>> pMap =
							service.getAllTutorialProviders(cat);
					for(String title : pMap.keySet()) {
						Class<? extends ITutorialProvider> pClass = pMap.get(title);
						catManager.add(new Action(title, imgDescr) {
							@Override
							public void run() {
								ITutorialService.getInstance().startTutorial(pClass);
							}
						});
					}
					manager.add(catManager);
				}
			}

			/** Creates menu item to stop currently running tutorial. */
			private void createStopTutorialItem(IMenuManager manager) {
				manager.add(new Action("Stop Current Tutorial") {
					/** {@inheritDoc} */
					@Override
					public void run() {
						ITutorialService.getInstance().stopTutorial();
					}
				});
			}
		});
		return tutorialMenu;
	}
}
