/*-------------------------------------------------------------------------+
| Copyright 2014 fortiss GmbH                                              |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package org.fortiss.tooling.kernel.ui.util;

import java.util.Stack;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.widgets.Text;

/**
 * Hack class to implement undo/redo key shortcuts for swt Text fields.
 * 
 * @author aravantinos
 */
public class UndoRedoImpl implements KeyListener, ModifyListener {
	/** Encapsulation of the generic undo and redo stack(s). */
	private static class UndoRedoStack<T> {

		/** Generic undo stack. */
		private Stack<T> undo;

		/** Generic redo stack. */
		private Stack<T> redo;

		/** Constructor. */
		public UndoRedoStack() {
			undo = new Stack<T>();
			redo = new Stack<T>();
		}

		/** Push on the undo stack. */
		public void pushUndo(T delta) {
			if(delta != null) {
				undo.add(delta);
			}
		}

		/** Push on the redo stack. */
		public void pushRedo(T delta) {
			if(delta != null) {
				redo.add(delta);
			}
		}

		/** Pop from the undo stack. */
		public T popUndo() {
			T res = undo.pop();
			return res;
		}

		/** Pop from the redo stack. */
		public T popRedo() {
			T res = redo.pop();
			return res;
		}

		/** Clear the redo stack. */
		public void clearRedo() {
			redo.clear();
		}

		/** Test if the undo stack is not empty. */
		public boolean hasUndo() {
			return !undo.isEmpty();
		}

		/** Test if the redo stack is not empty. */
		public boolean hasRedo() {
			return !redo.isEmpty();
		}
	}

	/**
	 * Class encapsulating a string and a cursor position. This is what is going to be stored in the
	 * {@link UndoRedoStack}.
	 */
	private static class StringAndPosition {
		/** String content. */
		public String string;

		/** Cursor position. */
		public int position;

		/** Constructor. */
		public StringAndPosition(String str, int pos) {
			string = str;
			position = pos;
		}
	}

	/** Text widget for which the undo/redo feature is implemented. */
	private Text editor;

	/** The undo/redo stack. */
	private UndoRedoStack<StringAndPosition> stack;

	/** Flag to know if we are in the course of applying an undo. */
	private boolean isUndo;

	/** Flag to know if we are in the course of applying a redo. */
	private boolean isRedo;

	/**
	 * To store the latest string and position (useful when we are in the course of applying an
	 * undo/redo and the current string and position in the widget do not represent a "valid"
	 * value).
	 */
	private StringAndPosition previousTextAndPosition;

	/**
	 * Creates a new instance of this class. Automatically starts listening to
	 * corresponding key and modify events coming from the given
	 * <var>editor</var>.
	 * 
	 * @param editor
	 *            the text field to which the Undo-Redo functionality should be
	 *            added
	 */
	public UndoRedoImpl(Text editor) {
		editor.addModifyListener(this);
		editor.addKeyListener(this);

		this.editor = editor;
		stack = new UndoRedoStack<StringAndPosition>();
	}

	/** Stores the information from the widget to the previousTextAndPosition attribute. */
	private void storeTextAndPos() {
		previousTextAndPosition = new StringAndPosition(editor.getText(), editor.getSelection().x);
	}

	/** Updates the text and cursor position of the widget according to the argument. */
	private void setTextAndPos(StringAndPosition x) {
		editor.setText(x.string);
		editor.setSelection(x.position);
	}

	/**
	 * We use keyPressed for Redo and keyReleased for Undo, because, for some reason, Ctrl-z is
	 * captured only at release and Ctrl-y (resp. Ctrl-Shift-y) only at press.
	 */
	@Override
	public void keyPressed(KeyEvent e) {
		storeTextAndPos();
		if((e.stateMask & SWT.MOD1) > 0 && !((e.stateMask & SWT.ALT) > 0)) {
			boolean isShift = (e.stateMask & SWT.SHIFT) > 0;
			if(!isShift && e.keyCode == 'y') {
				redo();
			}
		}
	}

	/**
	 * We use keyPressed for Redo and keyReleased for Undo, because, for some reason, Ctrl-z is
	 * captured only at release and Ctrl-y (resp. Ctrl-Shift-y) only at press.
	 */
	@Override
	public void keyReleased(KeyEvent e) {
		if((e.stateMask & SWT.MOD1) > 0 && !((e.stateMask & SWT.ALT) > 0)) {
			boolean isShift = (e.stateMask & SWT.SHIFT) > 0;
			if(!isShift && e.keyCode == 'z') {
				undo();
			} else if(isShift && e.keyCode == 'z') {
				redo();
			}
		}
	}

	/**
	 * Creates a corresponding Undo or Redo step from the given event and pushes
	 * it to the stack. The Redo stack is, logically, emptied if the event comes
	 * from a normal user action.
	 * 
	 * @param event
	 * @see org.eclipse.swt.custom.ExtendedModifyListener#modifyText(org.eclipse.
	 *      swt.custom.ExtendedModifyEvent)
	 */
	@Override
	public void modifyText(ModifyEvent event) {
		if(isUndo) {
			stack.pushRedo(previousTextAndPosition);
		} else { // is Redo or a normal user action
			stack.pushUndo(previousTextAndPosition);
			if(!isRedo) {
				stack.clearRedo();
			}
		}
	}

	/**
	 * Performs the Undo action. A new corresponding Redo step is automatically
	 * pushed to the stack.
	 */
	private void undo() {
		if(stack.hasUndo()) {
			storeTextAndPos();
			isUndo = true;
			setTextAndPos(stack.popUndo());
			isUndo = false;
		}
	}

	/**
	 * Performs the Redo action. A new corresponding Undo step is automatically
	 * pushed to the stack.
	 */
	private void redo() {
		if(stack.hasRedo()) {
			storeTextAndPos();
			isRedo = true;
			setTextAndPos(stack.popRedo());
			isRedo = false;
		}
	}
}
